MODULE rad_test_profiles

IMPLICIT NONE

INTEGER, PARAMETER :: RealK=SELECTED_REAL_KIND(15, 307)

INTEGER, PARAMETER :: n_lev = 50

REAL*8 :: &
    p_std(n_lev), &
    t_std(n_lev), &
    h2o_ppmv_std(n_lev), &
    co2_ppmv_std(n_lev), &
    o3_ppmv_std(n_lev), &
    n2o_ppmv_std(n_lev), &
    ch4_ppmv_std(n_lev)

! U.S. standard atmosphere 1976
REAL*8, PARAMETER :: &
    p_us_1976(n_lev) = (/ &
        1.013E+03, 8.988E+02, 7.950E+02, 7.012E+02, 6.166E+02, &
        5.405E+02, 4.722E+02, 4.111E+02, 3.565E+02, 3.080E+02, &
        2.650E+02, 2.270E+02, 1.940E+02, 1.658E+02, 1.417E+02, &
        1.211E+02, 1.035E+02, 8.850E+01, 7.565E+01, 6.467E+01, &
        5.529E+01, 4.729E+01, 4.047E+01, 3.467E+01, 2.972E+01, &
        2.549E+01, 1.743E+01, 1.197E+01, 8.010E+00, 5.746E+00, &
        4.150E+00, 2.871E+00, 2.060E+00, 1.491E+00, 1.090E+00, &
        7.978E-01, 4.250E-01, 2.190E-01, 1.090E-01, 5.220E-02, &
        2.400E-02, 1.050E-02, 4.460E-03, 1.840E-03, 7.600E-04, &
        3.200E-04, 1.450E-04, 7.100E-05, 4.010E-05, 2.540E-05  &
        /), &
!       Pressure [mbar]
    t_us_1976(n_lev) = (/ &
        288.20,    281.70,    275.20,    268.70,    262.20, &
        255.70,    249.20,    242.70,    236.20,    229.70, &
        223.30,    216.80,    216.70,    216.70,    216.70, &
        216.70,    216.70,    216.70,    216.70,    216.70, &
        216.70,    217.60,    218.60,    219.60,    220.60, &
        221.60,    224.00,    226.50,    230.00,    236.50, &
        242.90,    250.40,    257.30,    264.20,    270.60, &
        270.70,    260.80,    247.00,    233.30,    219.60, &
        208.40,    198.60,    188.90,    186.90,    188.40, &
        195.10,    208.80,    240.00,    300.00,    360.00  &
        /), &
!       Temperature
    h2o_ppmv_us_1976(n_lev) = (/ &
        7.745e+03, 6.071e+03, 4.631e+03, 3.182e+03, 2.158e+03, &
        1.397e+03, 9.254e+02, 5.720e+02, 3.667e+02, 1.583e+02, &
        6.996e+01, 3.613e+01, 1.906e+01, 1.085e+01, 5.927e+00, &
        5.000e+00, 3.950e+00, 3.850e+00, 3.825e+00, 3.850e+00, &
        3.900e+00, 3.975e+00, 4.065e+00, 4.200e+00, 4.300e+00, &
        4.425e+00, 4.575e+00, 4.725e+00, 4.825e+00, 4.900e+00, &
        4.950e+00, 5.025e+00, 5.150e+00, 5.225e+00, 5.250e+00, &
        5.225e+00, 5.100e+00, 4.750e+00, 4.200e+00, 3.500e+00, &
        2.825e+00, 2.050e+00, 1.330e+00, 8.500e-01, 5.400e-01, &
        4.000e-01, 3.400e-01, 2.800e-01, 2.400e-01, 2.000e-01  &
        /), &
!       H2O mixing ratio in PPMV
    co2_ppmv_us_1976(n_lev) = (/ &
        3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, &
        3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, &
        3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, &
        3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, &
        3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, &
        3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, &
        3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, &
        3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, 3.300e+02, &
        3.300e+02, 3.280e+02, 3.200e+02, 3.100e+02, 2.700e+02, &
        1.950e+02, 1.100e+02, 6.000e+01, 4.000e+01, 3.500e+01  &
        /), &
!       CO2 mixing ratio in PPMV
    o3_ppmv_us_1976(n_lev) = (/ &
        2.660e-02, 2.931e-02, 3.237e-02, 3.318e-02, 3.387e-02, &
        3.768e-02, 4.112e-02, 5.009e-02, 5.966e-02, 9.168e-02, &
        1.313e-01, 2.149e-01, 3.095e-01, 3.846e-01, 5.030e-01, &
        6.505e-01, 8.701e-01, 1.187e+00, 1.587e+00, 2.030e+00, &
        2.579e+00, 3.028e+00, 3.647e+00, 4.168e+00, 4.627e+00, &
        5.118e+00, 5.803e+00, 6.553e+00, 7.373e+00, 7.837e+00, &
        7.800e+00, 7.300e+00, 6.200e+00, 5.250e+00, 4.100e+00, &
        3.100e+00, 1.800e+00, 1.100e+00, 7.000e-01, 3.000e-01, &
        2.500e-01, 3.000e-01, 5.000e-01, 7.000e-01, 7.000e-01, &
        4.000e-01, 2.000e-01, 5.000e-02, 5.000e-03, 5.000e-04  &
        /), &
!       O3 mixing ratio in PPMV
    n2o_ppmv_us_1976(n_lev) = (/ &
        3.200e-01, 3.200e-01, 3.200e-01, 3.200e-01, 3.200e-01, &
        3.200e-01, 3.200e-01, 3.200e-01, 3.200e-01, 3.195e-01, &
        3.179e-01, 3.140e-01, 3.095e-01, 3.048e-01, 2.999e-01, &
        2.944e-01, 2.877e-01, 2.783e-01, 2.671e-01, 2.527e-01, &
        2.365e-01, 2.194e-01, 2.051e-01, 1.967e-01, 1.875e-01, &
        1.756e-01, 1.588e-01, 1.416e-01, 1.165e-01, 9.275e-02, &
        6.693e-02, 4.513e-02, 2.751e-02, 1.591e-02, 9.378e-03, &
        4.752e-03, 3.000e-03, 2.065e-03, 1.507e-03, 1.149e-03, &
        8.890e-04, 7.056e-04, 5.716e-04, 4.708e-04, 3.932e-04, &
        3.323e-04, 2.837e-04, 2.443e-04, 2.120e-04, 1.851e-04  &
        /), &
!       N2O mixing ratio in PPMV
    ch4_ppmv_us_1976(n_lev) = (/ &
        1.700e+00, 1.700e+00, 1.700e+00, 1.700e+00, 1.700e+00, &
        1.700e+00, 1.700e+00, 1.699e+00, 1.697e+00, 1.693e+00, &
        1.685e+00, 1.675e+00, 1.662e+00, 1.645e+00, 1.626e+00, &
        1.605e+00, 1.582e+00, 1.553e+00, 1.521e+00, 1.480e+00, &
        1.424e+00, 1.355e+00, 1.272e+00, 1.191e+00, 1.118e+00, &
        1.055e+00, 9.870e-01, 9.136e-01, 8.300e-01, 7.460e-01, &
        6.618e-01, 5.638e-01, 4.614e-01, 3.631e-01, 2.773e-01, &
        2.100e-01, 1.650e-01, 1.500e-01, 1.500e-01, 1.500e-01, &
        1.500e-01, 1.500e-01, 1.500e-01, 1.400e-01, 1.300e-01, &
        1.200e-01, 1.100e-01, 9.500e-02, 6.000e-02, 3.000e-02  &
        /)
!       CH4 mixing ratio in PPMV

! Midlatitude summer atmosphere
REAL*8, PARAMETER :: &
    p_midlat_summer(n_lev) = (/ &
        1.013E+03, 9.020E+02, 8.020E+02, 7.100E+02, 6.280E+02, &
        5.540E+02, 4.870E+02, 4.260E+02, 3.720E+02, 3.240E+02, &
        2.810E+02, 2.430E+02, 2.090E+02, 1.790E+02, 1.530E+02, &
        1.300E+02, 1.110E+02, 9.500E+01, 8.120E+01, 6.950E+01, &
        5.950E+01, 5.100E+01, 4.370E+01, 3.760E+01, 3.220E+01, &
        2.770E+01, 1.907E+01, 1.320E+01, 9.300E+00, 6.520E+00, &
        4.640E+00, 3.330E+00, 2.410E+00, 1.760E+00, 1.290E+00, &
        9.510E-01, 5.150E-01, 2.720E-01, 1.390E-01, 6.700E-02, &
        3.000E-02, 1.200E-02, 4.480E-03, 1.640E-03, 6.250E-04, &
        2.580E-04, 1.170E-04, 6.110E-05, 3.560E-05, 2.270E-05  &
        /), &
!       Pressure [mbar]
    t_midlat_summer(n_lev) = (/ &
        294.20,    289.70,    285.20,    279.20,    273.20, &
        267.20,    261.20,    254.70,    248.20,    241.70, &
        235.30,    228.80,    222.30,    215.80,    215.70, &
        215.70,    215.70,    215.70,    216.80,    217.90, &
        219.20,    220.40,    221.60,    222.80,    223.90, &
        225.10,    228.45,    233.70,    239.00,    245.20, &
        251.30,    257.50,    263.70,    269.90,    275.20, &
        275.70,    269.30,    257.10,    240.10,    218.10, &
        196.10,    174.10,    165.10,    165.00,    178.30, &
        190.50,    222.20,    262.40,    316.80,    380.00  &
        /), &
!       Temperature
    h2o_ppmv_midlat_summer(n_lev) = (/ &
        1.876E+04, 1.378E+04, 9.680E+03, 5.984E+03, 3.813E+03, &
        2.225E+03, 1.510E+03, 1.020E+03, 6.464E+02, 4.129E+02, &
        2.472E+02, 9.556E+01, 2.944E+01, 8.000E+00, 5.000E+00, &
        3.400E+00, 3.300E+00, 3.200E+00, 3.150E+00, 3.200E+00, &
        3.300E+00, 3.450E+00, 3.600E+00, 3.850E+00, 4.000E+00, &
        4.200E+00, 4.450E+00, 4.700E+00, 4.850E+00, 4.950E+00, &
        5.000E+00, 5.100E+00, 5.300E+00, 5.450E+00, 5.500E+00, &
        5.500E+00, 5.350E+00, 5.000E+00, 4.400E+00, 3.700E+00, &
        2.950E+00, 2.100E+00, 1.330E+00, 8.500E-01, 5.400E-01, &
        4.000E-01, 3.400E-01, 2.800E-01, 2.400E-01, 2.000E-01  &
        /), &
!       H2O mixing ratio in PPMV
    co2_ppmv_midlat_summer(n_lev) = (/ &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.280E+02, 3.200E+02, 3.100E+02, 2.700E+02, &
        1.950E+02, 1.100E+02, 6.000E+01, 4.000E+01, 3.500E+01  &
        /), &
!       CO2 mixing ratio in PPMV
    o3_ppmv_midlat_summer(n_lev) = (/ &
        3.017E-02, 3.337E-02, 3.694E-02, 4.222E-02, 4.821E-02, &
        5.512E-02, 6.408E-02, 7.764E-02, 9.126E-02, 1.111E-01, &
        1.304E-01, 1.793E-01, 2.230E-01, 3.000E-01, 4.400E-01, &
        5.000E-01, 6.000E-01, 7.000E-01, 1.000E+00, 1.500E+00, &
        2.000E+00, 2.400E+00, 2.900E+00, 3.400E+00, 4.000E+00, &
        4.800E+00, 6.000E+00, 7.000E+00, 8.100E+00, 8.900E+00, &
        8.700E+00, 7.550E+00, 5.900E+00, 4.500E+00, 3.500E+00, &
        2.800E+00, 1.800E+00, 1.300E+00, 8.000E-01, 4.000E-01, &
        1.900E-01, 2.000E-01, 5.700E-01, 7.500E-01, 7.000E-01, &
        4.000E-01, 2.000E-01, 5.000E-02, 5.000E-03, 5.000E-04  &
        /), &
!       O3 mixing ratio in PPMV
    n2o_ppmv_midlat_summer(n_lev) = (/ &
        3.200E-01, 3.200E-01, 3.200E-01, 3.200E-01, 3.200E-01, &
        3.200E-01, 3.200E-01, 3.200E-01, 3.195E-01, 3.163E-01, &
        3.096E-01, 2.989E-01, 2.936E-01, 2.860E-01, 2.800E-01, &
        2.724E-01, 2.611E-01, 2.421E-01, 2.174E-01, 1.843E-01, &
        1.607E-01, 1.323E-01, 1.146E-01, 1.035E-01, 9.622E-02, &
        8.958E-02, 8.006E-02, 6.698E-02, 4.958E-02, 3.695E-02, &
        2.519E-02, 1.736E-02, 1.158E-02, 7.665E-03, 5.321E-03, &
        3.215E-03, 2.030E-03, 1.397E-03, 1.020E-03, 7.772E-04, &
        6.257E-04, 5.166E-04, 4.352E-04, 3.727E-04, 3.237E-04, &
        2.844E-04, 2.524E-04, 2.260E-04, 2.039E-04, 1.851E-04  &
        /), &
!       N2O mixing ratio in PPMV
    ch4_ppmv_midlat_summer(n_lev) = (/ &
        1.700E+00, 1.700E+00, 1.700E+00, 1.700E+00, 1.697E+00, &
        1.687E+00, 1.672E+00, 1.649E+00, 1.629E+00, 1.615E+00, &
        1.579E+00, 1.542E+00, 1.508E+00, 1.479E+00, 1.451E+00, &
        1.422E+00, 1.390E+00, 1.356E+00, 1.323E+00, 1.281E+00, &
        1.224E+00, 1.154E+00, 1.066E+00, 9.730E-01, 8.800E-01, &
        7.888E-01, 7.046E-01, 6.315E-01, 5.592E-01, 5.008E-01, &
        4.453E-01, 3.916E-01, 3.389E-01, 2.873E-01, 2.384E-01, &
        1.944E-01, 1.574E-01, 1.500E-01, 1.500E-01, 1.500E-01, &
        1.500E-01, 1.500E-01, 1.500E-01, 1.400E-01, 1.300E-01, &
        1.200E-01, 1.100E-01, 9.500E-02, 6.000E-02, 3.000E-02  &
        /)
!       CH4 mixing ratio in PPMV

! Tropical atmosphere
REAL*8, PARAMETER :: &
    p_tropical(n_lev) = (/ &
        1.013E+03, 9.040E+02, 8.050E+02, 7.150E+02, 6.330E+02, &
        5.590E+02, 4.920E+02, 4.320E+02, 3.780E+02, 3.290E+02, &
        2.860E+02, 2.470E+02, 2.130E+02, 1.820E+02, 1.560E+02, &
        1.320E+02, 1.110E+02, 9.370E+01, 7.890E+01, 6.660E+01, &
        5.650E+01, 4.800E+01, 4.090E+01, 3.500E+01, 3.000E+01, &
        2.570E+01, 1.763E+01, 1.220E+01, 8.520E+00, 6.000E+00, &
        4.260E+00, 3.050E+00, 2.200E+00, 1.590E+00, 1.160E+00, &
        8.540E-01, 4.560E-01, 2.390E-01, 1.210E-01, 5.800E-02, &
        2.600E-02, 1.100E-02, 4.400E-03, 1.720E-03, 6.880E-04, &
        2.890E-04, 1.300E-04, 6.470E-05, 3.600E-05, 2.250E-05  &
        /), &
!       Pressure [mbar]
    t_tropical(n_lev) = (/ &
        299.70,    293.70,    287.70,    283.70,    277.00, &
        270.30,    263.60,    257.00,    250.30,    243.60, &
        237.00,    230.10,    223.60,    217.00,    210.30, &
        203.70,    197.00,    194.80,    198.80,    202.70, &
        206.70,    210.70,    214.60,    217.00,    219.20, &
        221.40,    227.00,    232.30,    237.70,    243.10, &
        248.50,    254.00,    259.40,    264.80,    269.60, &
        270.20,    263.40,    253.10,    236.00,    218.90, &
        201.80,    184.80,    177.10,    177.00,    184.30, &
        190.70,    212.00,    241.60,    299.70,    380.00  &
        /), &
!       Temperature
    h2o_ppmv_tropical(n_lev) = (/ &
        2.593E+04, 1.949E+04, 1.534E+04, 8.600E+03, 4.441E+03, &
        3.346E+03, 2.101E+03, 1.289E+03, 7.637E+02, 4.098E+02, &
        1.912E+02, 7.306E+01, 2.905E+01, 9.900E+00, 6.220E+00, &
        4.000E+00, 3.000E+00, 2.900E+00, 2.750E+00, 2.600E+00, &
        2.600E+00, 2.650E+00, 2.800E+00, 2.900E+00, 3.200E+00, &
        3.250E+00, 3.600E+00, 4.000E+00, 4.300E+00, 4.600E+00, &
        4.900E+00, 5.200E+00, 5.500E+00, 5.700E+00, 5.900E+00, &
        6.000E+00, 6.000E+00, 6.000E+00, 5.400E+00, 4.500E+00, &
        3.300E+00, 2.100E+00, 1.300E+00, 8.500E-01, 5.400E-01, &
        4.000E-01, 3.400E-01, 2.800E-01, 2.400E-01, 2.000E-01  &
        /), &
!       H2O mixing ratio in PPMV
    co2_ppmv_tropical(n_lev) = (/ &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, 3.300E+02, &
        3.300E+02, 3.280E+02, 3.200E+02, 3.100E+02, 2.700E+02, &
        1.950E+02, 1.100E+02, 6.000E+01, 4.000E+01, 3.500E+01  &
        /), &
!       CO2 mixing ratio in PPMV
    o3_ppmv_tropical(n_lev) = (/ &
        2.869E-02, 3.150E-02, 3.342E-02, 3.504E-02, 3.561E-02, &
        3.767E-02, 3.989E-02, 4.223E-02, 4.471E-02, 5.000E-02, &
        5.595E-02, 6.613E-02, 7.815E-02, 9.289E-02, 1.050E-01, &
        1.256E-01, 1.444E-01, 2.500E-01, 5.000E-01, 9.500E-01, &
        1.400E+00, 1.800E+00, 2.400E+00, 3.400E+00, 4.300E+00, &
        5.400E+00, 7.800E+00, 9.300E+00, 9.850E+00, 9.700E+00, &
        8.800E+00, 7.500E+00, 5.900E+00, 4.500E+00, 3.450E+00, &
        2.800E+00, 1.800E+00, 1.100E+00, 6.500E-01, 3.000E-01, &
        1.800E-01, 3.300E-01, 5.000E-01, 5.200E-01, 5.000E-01, &
        4.000E-01, 2.000E-01, 5.000E-02, 5.000E-03, 5.000E-04  &
        /), &
!       O3 mixing ratio in PPMV
    n2o_ppmv_tropical(n_lev) = (/ &
        3.200E-01, 3.200E-01, 3.200E-01, 3.200E-01, 3.200E-01, &
        3.200E-01, 3.200E-01, 3.200E-01, 3.200E-01, 3.195E-01, &
        3.179E-01, 3.140E-01, 3.095E-01, 3.048E-01, 2.999E-01, &
        2.944E-01, 2.877E-01, 2.783E-01, 2.671E-01, 2.527E-01, &
        2.365E-01, 2.194E-01, 2.051E-01, 1.967E-01, 1.875E-01, &
        1.756E-01, 1.588E-01, 1.416E-01, 1.165E-01, 9.275E-02, &
        6.693E-02, 4.513E-02, 2.751E-02, 1.591E-02, 9.378E-03, &
        4.752E-03, 3.000E-03, 2.065E-03, 1.507E-03, 1.149E-03, &
        8.890E-04, 7.056E-04, 5.716E-04, 4.708E-04, 3.932E-04, &
        3.323E-04, 2.837E-04, 2.443E-04, 2.120E-04, 1.851E-04  &
        /), &
!       N2O mixing ratio in PPMV
    ch4_ppmv_tropical(n_lev) = (/ &
        1.500E-01, 1.450E-01, 1.399E-01, 1.349E-01, 1.312E-01, &
        1.303E-01, 1.288E-01, 1.247E-01, 1.185E-01, 1.094E-01, &
        9.962E-02, 8.964E-02, 7.814E-02, 6.374E-02, 5.025E-02, &
        3.941E-02, 3.069E-02, 2.489E-02, 1.966E-02, 1.549E-02, &
        1.331E-02, 1.232E-02, 1.232E-02, 1.307E-02, 1.400E-02, &
        1.521E-02, 1.722E-02, 1.995E-02, 2.266E-02, 2.487E-02, &
        2.738E-02, 3.098E-02, 3.510E-02, 3.987E-02, 4.482E-02, &
        5.092E-02, 5.985E-02, 6.960E-02, 9.188E-02, 1.938E-01, &
        5.688E-01, 1.549E+00, 3.849E+00, 6.590E+00, 1.044E+01, &
        1.705E+01, 2.471E+01, 3.358E+01, 4.148E+01, 5.000E+01  &
        /)
!       CH4 mixing ratio in PPMV

CONTAINS

  SUBROUTINE select_atmosphere(atm_name)
  
    CHARACTER(LEN=*),INTENT(IN) :: atm_name
    
    IF (atm_name == 'us_standard_1976') THEN
      p_std = p_us_1976
      t_std = t_us_1976
      h2o_ppmv_std = h2o_ppmv_us_1976
      co2_ppmv_std = co2_ppmv_us_1976
      o3_ppmv_std = o3_ppmv_us_1976
      n2o_ppmv_std = n2o_ppmv_us_1976
      ch4_ppmv_std = ch4_ppmv_us_1976
    ELSE IF (atm_name == 'midlat_summer') THEN
      p_std = p_midlat_summer
      t_std = t_midlat_summer
      h2o_ppmv_std = h2o_ppmv_midlat_summer
      co2_ppmv_std = co2_ppmv_midlat_summer
      o3_ppmv_std = o3_ppmv_midlat_summer
      n2o_ppmv_std = n2o_ppmv_midlat_summer
      ch4_ppmv_std = ch4_ppmv_midlat_summer
    ELSE IF (atm_name == 'tropical') THEN
      p_std = p_tropical
      t_std = t_tropical
      h2o_ppmv_std = h2o_ppmv_tropical
      co2_ppmv_std = co2_ppmv_tropical
      o3_ppmv_std = o3_ppmv_tropical
      n2o_ppmv_std = n2o_ppmv_tropical
      ch4_ppmv_std = ch4_ppmv_tropical
    ELSE
      WRITE(0,*) 'Atmosphere not recognised.'
      STOP
    END IF
  
  END SUBROUTINE select_atmosphere

! Wrapper for the interpolation function.
  FUNCTION interp(x ,y, xi) RESULT(yi)

    IMPLICIT NONE

    REAL(RealK), INTENT(IN) ::                                          &
        x(:)                                                            &
!         Array with x-values
      , y(:)                                                            &
!         Array with y-values
      , xi
!         Value at which the y-coordinate is wanted

    REAL(RealK) ::                                                      &
        yi
!         Value at xi.

    yi = interp1(x, y, xi)


  END FUNCTION interp

! Liner interpolation function
  FUNCTION interp1(x ,y, xi) RESULT(yi)

    IMPLICIT NONE

    REAL(RealK), INTENT(IN) ::                                          &
        x(:)                                                            &
!         Array with x-values
      , y(:)                                                            &
!         Array with y-values
      , xi
!         Value at which the y-coordinate is wanted

    REAL(RealK) ::                                                      &
        yi
!         Value at xi.

    INTEGER ::                                                          &
        x_len                                                           &
!         Length of x-array
      , i
!         Loop index


    x_len = SIZE(x)

    IF (xi < x(1)) THEN
      yi = y(1)
      RETURN
    ELSE IF (xi > x(x_len)) THEN
      yi = y(x_len)
      RETURN
    ELSE
      DO i=2,x_len
        IF (xi <= x(i)) THEN
          yi = (y(i) - y(i-1))/(x(i) - x(i-1))*(xi - x(i-1)) + y(i-1)
          RETURN
        END IF
      END DO
    END IF

  END FUNCTION interp1

END MODULE rad_test_profiles