module fv_eta_mod
 use constants_mod,  only: kappa, grav, cp_air, rdgas
 use fv_mp_mod,      only: gid
 implicit none
 private
 public set_eta, get_eta_level, compute_dz_L32, set_hybrid_z, compute_dz, gw_1d

 contains

 subroutine set_eta(km, ks, ptop, ak, bk)

      integer,  intent(in)::  km           ! vertical dimension
      integer,  intent(out):: ks           ! number of pure p layers
      real, intent(out):: ak(km+1)
      real, intent(out):: bk(km+1)
      real, intent(out):: ptop         ! model top (Pa)
! local
      real a24(25),b24(25)            ! GFDL AM2L24
      real a26(27),b26(27)            ! Jablonowski & Williamson 26-level
      real a32(33),b32(33)
      real a32w(33),b32w(33)
      real a48(49),b48(49)
      real a55(56),b55(56)
      real a56(57),b56(57)            ! For Mars GCM
      real a64(65),b64(65)
      real a100(101),b100(101)


#ifdef MARS_GCM
      real:: p0= 6.0E2
      real:: pc= 1.0e-3
      real a36(37),b36(37)            ! For Mars GCM
      real a28(29),b28(29)            !  Mars 28 levels 
#else
      real:: p0=1000.E2
      real:: pc=200.E2
#endif MARS_GCM 

      real pt, pint, lnpe, dlnp
      real press(km+1), pt1(km)
      integer  k

#ifdef MARS_GCM
!------- Mars vertical levels -----------------------------

  data a28 /     &
       2.0000000e-02,5.7381273e-02,1.9583981e-01,5.9229580e-01, &
       1.5660228e+00,2.4454965e+00,2.7683755e+00,2.8851692e+00, &
       2.9172228e+00,2.9087039e+00,2.8598939e+00,2.7687652e+00, &
       2.6327054e+00,2.4509219e+00,2.2266811e+00,1.9684681e+00, &
       1.6894832e+00,1.4055812e+00,1.1324258e+00,8.8289177e-01, &
       6.6548467e-01,4.8401020e-01,3.3824119e-01,2.2510704e-01, &
       1.3995719e-01,7.7611554e-02,3.3085503e-02,2.0000000e-03, &
       0.0000000e+00  /

  data b28 /      &
       0.0000000e+00,0.0000000e+00,0.0000000e+00,0.0000000e+00, &
       0.0000000e+00,1.9366394e-03,7.4419133e-03,1.6227267e-02, &
       2.7075192e-02,4.3641000e-02,6.8106804e-02,1.0280240e-01, &
       1.4971954e-01,2.0987133e-01,2.8270233e-01,3.6581610e-01, &
       4.5520230e-01,5.4593599e-01,6.3310970e-01,7.1267629e-01, &
       7.8196151e-01,8.3977530e-01,8.8620345e-01,9.2223168e-01, &
       9.4934533e-01,9.6919618e-01,9.8337259e-01,9.9326941e-01, &
       1.0000000e+00  /


  data a36 /      &
  2.0000000000e-03,  6.3299931399e-03,  1.2501646444e-02, & 
  2.4215113043e-02,  4.6000346612e-02,  8.5702012910e-02, & 
  1.5659441036e-01,  2.8061882660e-01,  4.9318818941e-01, & 
  8.5008792314e-01,  1.4370449074e+00,  2.0054945771e+00, & 
  2.3335916338e+00,  2.5221957520e+00,  2.6264602874e+00, & 
  2.6762480591e+00,  2.6870173757e+00,  2.6657174771e+00, & 
  2.6140573091e+00,  2.5304052915e+00,  2.4110051828e+00, &
  2.2508885698e+00,  2.0446940567e+00,  1.8074358726e+00, & 
  1.5511877409e+00,  1.2904762231e+00,  1.0396626704e+00, & 
  8.1055438393e-01,  6.1095195817e-01,  4.4434410651e-01, & 
  3.1051916267e-01,  2.0665632614e-01,  1.2848513437e-01, & 
  7.1249514632e-02,  3.0373097709e-02,  2.1040298410e-16, & 
  0.0000000000e+00  /
  
  data b36 /      &
  0.0000000000e+00,  0.0000000000e+00,  0.0000000000e+00, & 
  0.0000000000e+00,  0.0000000000e+00,  0.0000000000e+00, & 
  0.0000000000e+00,  0.0000000000e+00,  0.0000000000e+00, & 
  0.0000000000e+00,  0.0000000000e+00,  6.2833541661e-04, & 
  2.5671934940e-03,  6.0923860938e-03,  1.1725248117e-02, & 
  2.0238695380e-02,  3.2676843179e-02,  5.0373898012e-02, & 
  7.4959525687e-02,  1.0833815470e-01,  1.5263019875e-01, & 
  2.1006506655e-01,  2.8281982789e-01,  3.6584598427e-01, & 
  4.5513763376e-01,  5.4577516959e-01,  6.3285639147e-01, & 
  7.1233851133e-01,  7.8155013540e-01,  8.3930250332e-01, & 
  8.8568132228e-01,  9.2167127115e-01,  9.4875609753e-01, & 
  9.6858585131e-01,  9.8274719475e-01,  9.9326941371e-01, & 
  1.0000000000e+00  /

#endif MARS_GCM 

! Definition: press(i,j,k) = ak(k) + bk(k) * ps(i,j)

!-----------------------------------------------
! GFDL AM2-L24: modified by SJL at the model top
!-----------------------------------------------
!     data a24 /  100.0000,  1050.0000,  3474.7942,  7505.5556, 12787.2428,   &
      data a24 /  100.0000,   903.4465,  3474.7942,  7505.5556, 12787.2428,   &
                19111.3683, 21854.9274, 22884.1866, 22776.3058, 21716.1604,   &
                20073.2963, 18110.5123, 16004.7832, 13877.6253, 11812.5452,   &
                 9865.8840,  8073.9726,  6458.0834,  5027.9899,  3784.6085,   &
                 2722.0086,  1828.9752,  1090.2396,   487.4595,     0.0000    /

      data b24 / 0.0000000, 0.0000000, 0.0000000, 0.0000000, 0.0000000,       &
                 0.0000000, 0.0435679, 0.1102275, 0.1922249, 0.2817656,       &
                 0.3694997, 0.4532348, 0.5316253, 0.6038733, 0.6695556,       &
                 0.7285176, 0.7808017, 0.8265992, 0.8662148, 0.9000406,       &
                 0.9285364, 0.9522140, 0.9716252, 0.9873523, 1.0000000        /

! Jablonowski & Williamson 26-level setup
      data a26 /  219.4067,   489.5209,   988.2418,  1805.2010,  2983.7240,  4462.3340,   &
                 6160.5870,  7851.2430,  7731.2710,  7590.1310,  7424.0860,   &
                 7228.7440,  6998.9330,  6728.5740,  6410.5090,  6036.3220,   &
                 5596.1110,  5078.2250,  4468.9600,  3752.1910,  2908.9490,   &
                  2084.739,   1334.443,    708.499,   252.1360,  0.0, 0.0     /

      data b26 / 0.0, 0.0, 0.0000000, 0.0000000, 0.0000000, 0.0000000, 0.0000000,&
                 0.0000000, 0.01505309, 0.03276228, 0.05359622, 0.07810627,      &
                 0.1069411, 0.1408637, 0.1807720, 0.2277220, 0.2829562,       &
                 0.3479364, 0.4243822, 0.5143168, 0.6201202, 0.7235355,       &
                 0.8176768, 0.8962153, 0.9534761, 0.9851122, 1.0000000        /


! High-resolution troposphere setup
! Revised Apr 14, 2004: PINT = 245.027 mb
      data a32/100.00000,     400.00000,     818.60211, &
              1378.88653,    2091.79519,    2983.64084, &
              4121.78960,    5579.22148,    7419.79300, &
              9704.82578,   12496.33710,   15855.26306, &
             19839.62499,   24502.73262,   28177.10152, &
             29525.28447,   29016.34358,   27131.32792, &
             24406.11225,   21326.04907,   18221.18357, &
             15275.14642,   12581.67796,   10181.42843, &
              8081.89816,    6270.86956,    4725.35001, &
              3417.39199,    2317.75459,    1398.09473, &
               632.49506,       0.00000,       0.00000  /

      data b32/0.00000,       0.00000,       0.00000, &
               0.00000,       0.00000,       0.00000, &
               0.00000,       0.00000,       0.00000, &
               0.00000,       0.00000,       0.00000, &
               0.00000,       0.00000,       0.01711, &
               0.06479,       0.13730,       0.22693, &
               0.32416,       0.42058,       0.51105, &
               0.59325,       0.66628,       0.73011, &
               0.78516,       0.83217,       0.87197, &
               0.90546,       0.93349,       0.95685, &
               0.97624,       0.99223,       1.00000  /

!---------------------
! Wilson's 32L settings:
!---------------------
! Top changed to 0.01 mb
      data a32w/  1.00,       26.6378,     84.5529,     228.8592,   & 
                539.9597,   1131.7087,   2141.8082,    3712.0454,   &
               5963.5317,   8974.1873,  12764.5388,   17294.5911,   &
              20857.7007,  22221.8651,  22892.7202,   22891.1641,   &
              22286.0724,  21176.0846,  19673.0671,   17889.0989,   &
              15927.5060,  13877.6239,  11812.5474,    9865.8830,   &
               8073.9717,   6458.0824,   5027.9893,    3784.6104,   &
               2722.0093,   1828.9741,   1090.2397,     487.4575,   &
               0.0000 /
         
      data b32w/ 0.0000,   0.0000,   0.0000,    0.0000,       &
                0.0000,   0.0000,   0.0000,    0.0000,       &
                0.0000,   0.0000,   0.0000,    0.0000,       &
                0.0159,   0.0586,   0.1117,    0.1734,       &
                0.2415,   0.3137,   0.3878,    0.4619,       &
                0.5344,   0.6039,   0.6696,    0.7285,       &
                0.7808,   0.8266,   0.8662,    0.9000,       &
                0.9285,   0.9522,   0.9716,    0.9874,       &
                1.0000 /

! High trop-resolution (Feb 2004)
           data a48/                                &
                   1.00000,       2.69722,       5.17136,   &
                   8.89455,      14.24790,      22.07157,   &
                  33.61283,      50.48096,      74.79993,   &
                 109.40055,     158.00460,     225.44108,   &
                 317.89560,     443.19350,     611.11558,   &
                 833.74392,    1125.83405,    1505.20759,   &
                1993.15829,    2614.86254,    3399.78420,   &
                4382.06240,    5600.87014,    7100.73115,   &
                8931.78242,   11149.97021,   13817.16841,   &
               17001.20930,   20775.81856,   23967.33875,   &
               25527.64563,   25671.22552,   24609.29622,   &
               22640.51220,   20147.13482,   17477.63530,   &
               14859.86462,   12414.92533,   10201.44191,   &
                8241.50255,    6534.43202,    5066.17865,   &
                3815.60705,    2758.60264,    1870.64631,   &
                1128.33931,     510.47983,       0.00000,   &
                   0.00000  /

           data b48/              &
                   0.00000,       0.00000,       0.00000,   &
                   0.00000,       0.00000,       0.00000,   &
                   0.00000,       0.00000,       0.00000,   &
                   0.00000,       0.00000,       0.00000,   &
                   0.00000,       0.00000,       0.00000,   &
                   0.00000,       0.00000,       0.00000,   &
                   0.00000,       0.00000,       0.00000,   &
                   0.00000,       0.00000,       0.00000,   &
                   0.00000,       0.00000,       0.00000,   &
                   0.00000,       0.00000,       0.01253,   &
                   0.04887,       0.10724,       0.18455,   &
                   0.27461,       0.36914,       0.46103,   &
                   0.54623,       0.62305,       0.69099,   &
                   0.75016,       0.80110,       0.84453,   &
                   0.88127,       0.91217,       0.93803,   &
                   0.95958,       0.97747,       0.99223,   &
                   1.00000   /

      data a55/ 1.00000,     2.00000,       3.27000,       &
              4.75850,       6.60000,       8.93450,       &
             11.97030,      15.94950,      21.13490,       &
             27.85260,      36.50410,      47.58060,       &
             61.67790,      79.51340,     101.94420,       &
            130.05080,     165.07920,     208.49720,       &
            262.02120,     327.64330,     407.65670,       &
            504.68050,     621.68000,     761.98390,       &
            929.29430,    1127.68880,    1364.33920,       &
           1645.70720,    1979.15540,    2373.03610,       &
           2836.78160,    3380.99550,    4017.54170,       &
           4764.39320,    5638.79380,    6660.33770,       &
           7851.22980,    9236.56610,   10866.34270,       &
          12783.70000,   15039.30000,   17693.00000,       &
          20119.20876,   21686.49129,   22436.28749,       &
          22388.46844,   21541.75227,   19873.78342,       &
          17340.31831,   13874.44006,   10167.16551,       &
           6609.84274,    3546.59643,    1270.49390,       &
              0.00000,       0.00000   /

      data b55 /0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     & 
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00696,       0.02801,       0.06372,     &
                0.11503,       0.18330,       0.27033,     &
                0.37844,       0.51046,       0.64271,     &
                0.76492,       0.86783,       0.94329,     &
                0.98511,       1.00000  /

! The 56-L setup Can be used for Mars:
      data a56/ 0.33, 1.00000,  2.00000,    3.27000,       &
              4.75850,       6.60000,       8.93450,       &
             11.97030,      15.94950,      21.13490,       &
             27.85260,      36.50410,      47.58060,       &
             61.67790,      79.51340,     101.94420,       &
            130.05080,     165.07920,     208.49720,       &
            262.02120,     327.64330,     407.65670,       &
            504.68050,     621.68000,     761.98390,       &
            929.29430,    1127.68880,    1364.33920,       &
           1645.70720,    1979.15540,    2373.03610,       &
           2836.78160,    3380.99550,    4017.54170,       &
           4764.39320,    5638.79380,    6660.33770,       &
           7851.22980,    9236.56610,   10866.34270,       &
          12783.70000,   15039.30000,   17693.00000,       &
          20119.20876,   21686.49129,   22436.28749,       &
          22388.46844,   21541.75227,   19873.78342,       &
          17340.31831,   13874.44006,   10167.16551,       &
           6609.84274,    3546.59643,    1270.49390,       &
              0.00000,       0.00000   /

      data b56 /0., 0.00000,       0.00000,       0.00000, &
                0.00000,       0.00000,       0.00000,     & 
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00000,       0.00000,       0.00000,     &
                0.00696,       0.02801,       0.06372,     &
                0.11503,       0.18330,       0.27033,     &
                0.37844,       0.51046,       0.64271,     &
                0.76492,       0.86783,       0.94329,     &
                0.98511,       1.00000  /

      data a64/1.00000,       3.90000,       8.70000,      &
              15.42000,      24.00000,      34.50000,      &
              47.00000,      61.50000,      78.60000,      &
              99.13500,     124.12789,     154.63770,      &
             191.69700,     236.49300,     290.38000,      &
             354.91000,     431.82303,     523.09300,      &
             630.92800,     757.79000,     906.45000,      &
            1079.85000,    1281.00000,    1515.00000,      &
            1788.00000,    2105.00000,    2470.00000,      &
            2889.00000,    3362.00000,    3890.00000,      &
            4475.00000,    5120.00000,    5830.00000,      &
            6608.00000,    7461.00000,    8395.00000,      &
            9424.46289,   10574.46880,   11864.80270,      &
           13312.58890,   14937.03710,   16759.70700,      &
           18804.78710,   21099.41210,   23674.03710,      &
           26562.82810,   29804.11720,   32627.31640,      &
           34245.89840,   34722.28910,   34155.19920,      &
           32636.50390,   30241.08200,   27101.44920,      &
           23362.20700,   19317.05270,   15446.17090,      &
           12197.45210,    9496.39941,    7205.66992,      &
            5144.64307,    3240.79346,    1518.62134,      &
               0.00000,       0.00000 /

      data b64/0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00000,      &
               0.00000,       0.00000,       0.00813,      &
               0.03224,       0.07128,       0.12445,      &
               0.19063,       0.26929,       0.35799,      &
               0.45438,       0.55263,       0.64304,      &
               0.71703,       0.77754,       0.82827,      &
               0.87352,       0.91502,       0.95235,      &
               0.98511,       1.00000 /
!
! Ultra high troposphere resolution
      data a100/100.00000,     300.00000,     800.00000,   &
               1762.35235,    3106.43596,    4225.71874,   &
               4946.40525,    5388.77387,    5708.35540,   &
               5993.33124,    6277.73673,    6571.49996,   &
               6877.05339,    7195.14327,    7526.24920,   &
               7870.82981,    8229.35361,    8602.30193,   &
               8990.16936,    9393.46399,    9812.70768,   &
              10248.43625,   10701.19980,   11171.56286,   &
              11660.10476,   12167.41975,   12694.11735,   &
              13240.82253,   13808.17600,   14396.83442,   &
              15007.47066,   15640.77407,   16297.45067,   &
              16978.22343,   17683.83253,   18415.03554,   &
              19172.60771,   19957.34218,   20770.05022,   &
              21559.14829,   22274.03147,   22916.87519,   &
              23489.70456,   23994.40187,   24432.71365,   &
              24806.25734,   25116.52754,   25364.90190,   &
              25552.64670,   25680.92203,   25750.78675,   &
              25763.20311,   25719.04113,   25619.08274,   &
              25464.02630,   25254.49482,   24991.06137,   &
              24674.32737,   24305.11235,   23884.79781,   &
              23415.77059,   22901.76510,   22347.84738,   &
              21759.93950,   21144.07284,   20505.73136,   &
              19849.54271,   19179.31412,   18498.23400,   &
              17809.06809,   17114.28232,   16416.10343,   &
              15716.54833,   15017.44246,   14320.43478,   &
              13627.01116,   12938.50682,   12256.11762,   &
              11580.91062,   10913.83385,   10255.72526,   &
               9607.32122,    8969.26427,    8342.11044,   &
               7726.33606,    7122.34405,    6530.46991,   &
               5950.98721,    5384.11279,    4830.01153,   &
               4288.80090,    3760.55514,    3245.30920,   &
               2743.06250,    2253.78294,    1777.41285,   &
               1313.88054,     863.12371,     425.13088,   &
                  0.00000,       0.00000  /


      data b100/0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00000,       0.00000,       0.00000,   &
                0.00052,       0.00209,       0.00468,   &
                0.00828,       0.01288,       0.01849,   &
                0.02508,       0.03266,       0.04121,   &
                0.05075,       0.06126,       0.07275,   &
                0.08521,       0.09866,       0.11308,   &
                0.12850,       0.14490,       0.16230,   &
                0.18070,       0.20009,       0.22042,   &
                0.24164,       0.26362,       0.28622,   &
                0.30926,       0.33258,       0.35605,   &
                0.37958,       0.40308,       0.42651,   &
                0.44981,       0.47296,       0.49591,   &
                0.51862,       0.54109,       0.56327,   &
                0.58514,       0.60668,       0.62789,   &
                0.64872,       0.66919,       0.68927,   &
                0.70895,       0.72822,       0.74709,   &
                0.76554,       0.78357,       0.80117,   &
                0.81835,       0.83511,       0.85145,   &
                0.86736,       0.88286,       0.89794,   &
                0.91261,       0.92687,       0.94073,   &
                0.95419,       0.96726,       0.97994,   &
                0.99223,       1.00000  /
      select case (km)

       case (24)

          ks = 5     
          do k=1,km+1
            ak(k) = a24(k)
            bk(k) = b24(k)
          enddo

       case (26)
                
          ks = 7
          do k=1,km+1
            ak(k) = a26(k)     
            bk(k) = b26(k)     
          enddo

        case (32)
!         ks = 11              ! John Wilson's setup
          ks = 13              ! high-res trop_32 setup
          do k=1,km+1
            ak(k) = a32(k)
            bk(k) = b32(k)
          enddo

#ifdef MARS_GCM
       case (28)
          ks = 4     
          do k=1,km+1
            ak(k) = a28(k)
            bk(k) = b28(k)
          enddo

       case (36)
          ks = 10     
          do k=1,km+1
            ak(k) = a36(k)
            bk(k) = b36(k)
          enddo
#endif MARS_GCM 

        case (48)
          ks = 28
          do k=1,km+1
            ak(k) = a48(k)
            bk(k) = b48(k)
          enddo

        case (55)
          ks = 41
          do k=1,km+1
            ak(k) = a55(k)
            bk(k) = b55(k)
          enddo

        case (56)
          ks = 42
          do k=1,km+1
#ifdef MARS_GCM
            ak(k) = a56(k) * (6./1000.)
#else
            ak(k) = a56(k)
#endif
            bk(k) = b56(k)
          enddo

        case (64)
          ks = 46
          do k=1,km+1
            ak(k) = a64(k)
            bk(k) = b64(k)
          enddo

        case (100)
          ks = 38
          do k=1,km+1
            ak(k) = a100(k)
            bk(k) = b100(k)
          enddo

#ifndef TEST_GWAVES
        case (10)
!--------------------------------------------------
! Pure sigma-coordinate with uniform spacing in "z"
!--------------------------------------------------
!
         pt = 2000.           ! model top pressure (pascal)
!        pt =  100.           ! 1 mb
         press(1) = pt
         press(km+1) = p0
         dlnp = (log(p0) - log(pt)) / real(km)

            lnpe = log(press(km+1))
         do k=km,2,-1
            lnpe = lnpe - dlnp
            press(k) = exp(lnpe)
         enddo

! Search KS
            ks = 0
         do k=1,km
            if(press(k) >= pc) then
               ks = k-1
               goto 123
            endif
         enddo
123   continue

         if(ks /= 0) then
            do k=1,ks
               ak(k) = press(k)
               bk(k) = 0.
            enddo                                                
          endif                                                

             pint = press(ks+1)
          do k=ks+1,km                                        
             ak(k) =  pint*(press(km)-press(k))/(press(km)-pint)               
             bk(k) = (press(k) - ak(k)) / press(km+1)          
          enddo                                                
             ak(km+1) = 0. 
             bk(km+1) = 1.                                     
                                                              
!         do k=2,km
!            bk(k) = real(k-1) / real(km)
!            ak(k) = pt * ( 1. - bk(k) )
!         enddo

             ak(km+1) = 0.
             bk(km+1) = 1.
#endif

        case default

#ifdef TEST_GWAVES
!--------------------------------------------------
! Pure sigma-coordinate with uniform spacing in "z"
!--------------------------------------------------
          call gw_1d(km, 1000.E2, ak, bk, ptop, 10.E3, pt1)

            ks = 0
          pint = ak(1)
#else

!----------------------------------------------------------------
! Sigma-coordinate with uniform spacing in sigma and ptop = 1 mb
!----------------------------------------------------------------
         pt = 100.
! One pressure layer
         ks = 1
         pint = pt + 0.5*1.E5/real(km)

         ak(1) = pt
         bk(1) = 0.
         ak(2) = pint
         bk(2) = 0.
 
          do k=3,km+1
             bk(k) = real(k-2) / real(km-1)
             ak(k) = pint - bk(k)*pint
          enddo
          ak(km+1) = 0.
          bk(km+1) = 1.
#endif
      end select
      ptop = ak(1)

 end subroutine set_eta


 subroutine get_eta_level(npz, p_s, pf, ph, ak, bk, pscale)
  integer, intent(in) :: npz    
  real, intent(in)  :: p_s            ! unit: pascal
  real, intent(in)  :: ak(npz+1)
  real, intent(in)  :: bk(npz+1)
  real, intent(in), optional :: pscale
  real, intent(out) :: pf(npz)
  real, intent(out) :: ph(npz+1)
  integer k

  ph(1) = ak(1)               
  do k=2,npz+1
     ph(k) = ak(k) + bk(k)*p_s
  enddo                           
   
  if ( present(pscale) ) then
      do k=1,npz+1
         ph(k) = pscale*ph(k)
      enddo
  endif 

  if( ak(1) > 1.E-8 ) then   
     pf(1) = (ph(2) - ph(1)) / log(ph(2)/ph(1))
  else
     pf(1) = (ph(2) - ph(1)) * kappa/(kappa+1.)
  endif

  do k=2,npz
     pf(k) = (ph(k+1) - ph(k)) / log(ph(k+1)/ph(k))
  enddo

 end subroutine get_eta_level



 subroutine compute_dz(km, ztop, dz)

  integer, intent(in):: km
  real,   intent(in):: ztop        ! try 50.E3
  real,   intent(out):: dz(km)
!------------------------------
  real ze(km+1), dzt(km)
  integer k


! ztop = 30.E3
  dz(1) = ztop / real(km) 
  dz(km) = 0.5*dz(1)

  do k=2,km-1
     dz(k) = dz(1)
  enddo

! Top:
  dz(1) = 2.*dz(2)

  ze(km+1) = 0.
  do k=km,1,-1
     ze(k) = ze(k+1) + dz(k)
  enddo

  if ( gid==0 ) then
       write(*,*) 'Hybrid_z:  dz, zm'
       do k=1,km
          dzt(k) = 0.5*(ze(k)+ze(k+1)) / 1000.
          write(*,*) k, dz(k), dzt(k)
       enddo
  endif

 end subroutine compute_dz

 subroutine compute_dz_L32(km, ztop, dz)

  integer, intent(in):: km
  real,   intent(out):: dz(km)
  real,   intent(out):: ztop        ! try 50.E3
!------------------------------
  real dzt(km)
  real ze(km+1)
  real dz0, dz1, dz2
  real z0, z1, z2
  integer k, k0, k1, k2, n

!-------------------
        k2 =  8
        z2 = 30.E3
!-------------------
        k1 = 21
        z1 = 10.0E3
!-------------------
        k0 = 2
        z0 = 0.
!       dz0 = 80.   ! meters
        dz0 = 75.   ! meters
!-------------------
! Treat the surface layer as a special layer
        ze(1) = z0
        dz(1) = dz0

        ze(2) = dz(1)
          dz0 = 1.5*dz0
        dz(2) = dz0     

        ze(3) = ze(2) + dz(2)

        dz1 = 2.*(z1-ze(3) - k1*dz0) / (k1*(k1-1))

        do k=k0+1,k0+k1
           dz(k) = dz0 + (k-k0)*dz1
           ze(k+1) = ze(k) + dz(k)
        enddo

        dz0 = dz(k1+k0)
        dz2 = 2.*(z2-ze(k0+k1+1)-k2*dz0) / (k2*(k2-1))

        do k=k0+k1+1,k0+k1+k2
           dz(k) = dz0 + (k-k0-k1)*dz2
           ze(k+1) = ze(k) + dz(k)
        enddo

        dz(km) = 2.*dz(km-1)
        ztop = ze(km) + dz(km)
        ze(km+1) = ze(km) + dz(km)

        call zflip (dz, 1, km)

        ze(km+1) = 0.
        do k=km,1,-1
           ze(k) = ze(k+1) + dz(k)
        enddo

        if ( gid==0 ) then
           write(*,*) 'Hybrid_z:  dz, zm'
           do k=1,km
              dzt(k) = 0.5*(ze(k)+ze(k+1)) / 1000.
              write(*,*) k, dz(k), dzt(k)
           enddo
        endif

 end subroutine compute_dz_L32


 subroutine set_hybrid_z(is, ie, js, je, ng, km, ztop, dz, rgrav, hs, ze, delz)

 integer,  intent(in):: is, ie, js, je, ng, km
 real, intent(in):: rgrav, ztop
 real, intent(in):: dz(km)       ! Reference vertical resolution for zs=0
 real, intent(in):: hs(is-ng:ie+ng,js-ng:je+ng)
 real, intent(out), optional:: delz(is:ie,js:je,km)
 real, intent(out)::   ze(is:ie,js:je,km+1)
! local
 integer ntimes
 real zint
 real:: z1(is:ie,js:je)
 real:: z(km+1)
 real sig
 integer ks(is:ie,js:je)
 integer i, j, k, ks_min, kint

 z(km+1) = 0.
 do k=km,1,-1
    z(k) = z(k+1) + dz(k)
 enddo

  do j=js,je
     do i=is,ie
        ze(i,j,   1) = ztop 
        ze(i,j,km+1) = hs(i,j) * rgrav 
     enddo
  enddo

 do k=2,km
   do j=js,je
      do i=is,ie
         ze(i,j,k) = z(k)
      enddo
   enddo
 enddo

! Set interface:
#ifdef USE_CONST_ZINT
  zint = 8.5E3
  ntimes = 4
  kint = 2
  do k=2,km
     if ( z(k)<=zint ) then
          kint = k
          exit
     endif
  enddo

  if ( gid==0 ) write(*,*) 'Z_coord interface set at k=',kint, ' ZE=', z(kint)

  do j=js,je
     do i=is,ie
        do k=kint+1,km
           sig = 1. - min(1., z(k)/z(kint))
           ze(i,j,k) = z(k) + ze(i,j,km+1)*sig
        enddo
!--------------------------------------
! Apply vertical smoother locally to dz
!--------------------------------------
        call sm1_edge(is, ie, js, je, km, i, j, ze, ntimes)
    enddo
  enddo
#else
! ZINT is a function of local terrain
  ntimes = 5
  do j=js,je
     do i=is,ie
        z1(i,j) = 1.75*max(0.0, ze(i,j,km+1)) + z(km-2)
     enddo
  enddo

   ks_min = km
   do j=js,je
      do i=is,ie
         do k=km,2,-1
            if ( z(k)>=z1(i,j) ) then
                 ks(i,j) = k
                 ks_min = min(ks_min, k)
                 go to 555
            endif
        enddo
555     continue
      enddo
   enddo

  do j=js,je
     do i=is,ie
        kint = ks(i,j) + 1
        do k=kint,km
           sig = 1. - min(1., z(k)/z1(i,j))
           ze(i,j,k) = z(k) + ze(i,j,km+1)*sig
        enddo
!--------------------------------------
! Apply vertical smoother locally to dz
!--------------------------------------
        call sm1_edge(is, ie, js, je, km, i, j, ze, ntimes)
    enddo
  enddo
#endif

  if ( present(delz) ) then
  do k=1,km
     do j=js,je
        do i=is,ie
           delz(i,j,k) = ze(i,j,k+1) - ze(i,j,k)
           if ( delz(i,j,k) > 0. ) then
                write(*,*) 'Error in set_hybrid_z:', k,j,i, delz(i,j,k)
!               stop
           endif
        enddo
     enddo
  enddo
  endif

  end subroutine set_hybrid_z


  subroutine sm1_edge(is, ie, js, je, km, i, j, ze, ntimes)
  integer, intent(in):: is, ie, js, je, km
  integer, intent(in):: ntimes, i, j
  real, intent(inout):: ze(is:ie,js:je,km+1)
! local:
  real, parameter:: df = 0.25
  real dz(km)
  real flux(km+1)
  integer k, n, k1, k2

      k2 = km-1
      do k=1,km
         dz(k) = ze(i,j,k+1) - ze(i,j,k)
      enddo

   do n=1,ntimes
      k1 = 2 + (ntimes-n)

      flux(k1  ) = 0.
      flux(k2+1) = 0.
      do k=k1+1,k2
         flux(k) = df*(dz(k) - dz(k-1))
      enddo

      do k=k1,k2
         dz(k) = dz(k) - flux(k) + flux(k+1)
      enddo
   enddo

   do k=km,1,-1
      ze(i,j,k) = ze(i,j,k+1) - dz(k)
   enddo

  end subroutine sm1_edge

  subroutine gw_1d(km, p0, ak, bk, ptop, ztop, pt1)
  integer, intent(in):: km
  real,    intent(in):: p0, ztop
  real,    intent(inout):: ptop
  real,    intent(inout):: ak(km+1), bk(km+1)
  real, intent(out):: pt1(km)
! Local
  logical:: isothermal
  real, dimension(km+1):: ze, pe1, pk1
  real, dimension(km):: dz1
  real t0, n2, s0
  integer  k

! Set up vertical coordinare with constant del-z spacing:
       isothermal = .true.
       t0 = 300.

       if ( isothermal ) then
            n2 = grav**2/(cp_air*t0)
       else
            n2 = 0.0001
       endif

       s0 = grav*grav / (cp_air*n2) 

       ze(km+1) = 0.
       do k=km,1,-1
          dz1(k) = ztop / real(km)
           ze(k) = ze(k+1) + dz1(k)
       enddo

! Given z --> p
       do k=1,km+1
          pe1(k) = p0*( (1.-s0/t0) + s0/t0*exp(-n2*ze(k)/grav) )**(1./kappa)
       enddo

       ptop = pe1(1) 
!      if ( gid==0 ) write(*,*) 'GW_1D: computed model top (pa)=', ptop

! Set up "sigma" coordinate 
       ak(1) = pe1(1)
       bk(1) = 0.
       do k=2,km
          bk(k) = (pe1(k) - pe1(1)) / (pe1(km+1)-pe1(1))  ! bk == sigma
          ak(k) =  pe1(1)*(1.-bk(k)) 
       enddo                                                
       ak(km+1) = 0.
       bk(km+1) = 1.

       do k=1,km+1
          pk1(k) = pe1(k) ** kappa
       enddo

! Compute volume mean potential temperature with hydrostatic eqn:
       do k=1,km
          pt1(k) = grav*dz1(k) / ( cp_air*(pk1(k+1)-pk1(k)) )
       enddo

  end subroutine gw_1d



  subroutine zflip(q, im, km)
  integer, intent(in):: im, km
  real, intent(inout):: q(im,km)
!---
  integer i, k
  real qtmp

    do i = 1, im
       do k = 1, (km+1)/2
          qtmp = q(i,k)
          q(i,k) = q(i,km+1-k)
          q(i,km+1-k) = qtmp
       end do                                              
    end do                                                
                                                              
  end subroutine zflip   

end module fv_eta_mod
