module fv_mapz_mod

  use constants_mod, only: radius, pi, rvgas, rdgas, grav
  use fv_grid_tools_mod,    only: area, dx, dy, rdxa, rdya
  use fv_grid_utils_mod,    only: cubed_to_latlon, g_sum, ptop, ptop_min
  use fv_fill_mod,   only: fillz
  use fv_mp_mod,        only: gid, domain
  use mpp_domains_mod, only: mpp_update_domains

  implicit none
#if defined(TAF_DOES_NOT_LIKE)
  real, parameter::  r3 = 1./3., r23 = 2./3., r12 = 1./12.
#endif
  real*4 :: E_FLUX
  private

  public compute_total_energy, Lagrangian_to_Eulerian,    &
         rst_remap, mappm, E_Flux

CONTAINS

 subroutine Lagrangian_to_Eulerian(consv, ps, pe, delp, pkz, pk,   &
                      mdt, km, is,ie,js,je, isd,ied,jsd,jed,       &
                      nq, sphum, u, v, w, delz, pt, q, hs, r_vir, cp,  &
                      akap, kord_mt, kord_tr, kord_tm,  peln, te0_2d,        &
                      ng, ua, va, omga, te, pem, fill, reproduce_sum,        &
                      ak, bk, ks, ze0, remap_t, hydrostatic, hybrid_z, ktop, &
                      ncnst, mfx, mfy)
! !INPUT PARAMETERS:
  real,    intent(in):: mdt                   ! mapping time step (same as phys)
  integer, intent(in):: km
  integer, intent(in):: nq                    ! number of tracers (including h2o)
  integer, intent(in):: sphum                 ! index for water vapor (specific humidity)
  integer, intent(in):: ng
  integer, intent(in):: is,ie,isd,ied         ! starting & ending X-Dir index
  integer, intent(in):: js,je,jsd,jed         ! starting & ending Y-Dir index
  integer, intent(in):: ks, ktop
  integer, intent(in):: kord_mt               ! Mapping oder for the vector winds
  integer, intent(in):: kord_tr               ! Mapping oder for tracers
  integer, intent(in):: kord_tm               ! Mapping oder for thermodynamics
  integer, intent(in):: ncnst                 ! Added 4TAF

  real, intent(in):: consv                 ! factor for TE conservation
  real, intent(in):: r_vir
  real, intent(in):: cp
  real, intent(in):: akap
  real, intent(in):: hs(isd:ied,jsd:jed)  ! surface geopotential
  real, intent(in):: te0_2d(is:ie,js:je)

  logical, intent(in):: fill                  ! fill negative tracers
  logical, intent(in):: reproduce_sum
  real, intent(in) :: ak(km+1)
  real, intent(in) :: bk(km+1)

! !INPUT/OUTPUT
  real, intent(inout):: pk(is:ie,js:je,km+1) ! pe to the kappa
  real, intent(inout):: q(isd:ied,jsd:jed,km,ncnst)
  real, intent(inout):: delp(isd:ied,jsd:jed,km) ! pressure thickness
  real, intent(inout)::  pe(is-1:ie+1,km+1,js-1:je+1) ! pressure at layer edges
  real, intent(inout):: pem(is-1:ie+1,km+1,js-1:je+1)
  real, intent(inout):: ps(isd:ied,jsd:jed)      ! surface pressure
  real, intent(inout):: ze0(is:ie,js:je,km+1)    ! Specified height at edges (m)

! u-wind will be ghosted one latitude to the north upon exit
  real, intent(inout)::  u(isd:ied  ,jsd:jed+1,km)   ! u-wind (m/s)
  real, intent(inout)::  v(isd:ied+1,jsd:jed  ,km)   ! v-wind (m/s)
  real, intent(inout)::  w(isd:ied  ,jsd:jed  ,km)   ! vertical velocity (m/s)
  real, intent(inout):: pt(isd:ied  ,jsd:jed  ,km)   ! cp*virtual potential temperature 
                                                     ! as input; output: temperature
  real, intent(inout):: delz(is:ie,js:je,km)   ! delta-height (m)
  logical, intent(in):: remap_t
  logical, intent(in):: hydrostatic
  logical, intent(in):: hybrid_z

  real, intent(inout)::   ua(isd:ied,jsd:jed,km)   ! u-wind (m/s) on physics grid
  real, intent(inout)::   va(isd:ied,jsd:jed,km)   ! v-wind (m/s) on physics grid
  real, intent(inout):: omga(isd:ied,jsd:jed,km)   ! vertical press. velocity (pascal/sec)
  real, intent(inout)::   peln(is:ie,km+1,js:je)     ! log(pe)
  real, intent(out)::    pkz(is:ie,js:je,km)       ! layer-mean pk for converting t to pt
  real, intent(out)::     te(is:ie,js:je,km)

! Mass fluxes
  real, optional, intent(inout):: mfx(is:ie+1,js:je  ,km)   ! X-dir Mass Flux
  real, optional, intent(inout):: mfy(is:ie  ,js:je+1,km)   ! Y-dir Mass Flux

! !DESCRIPTION:
!
! !REVISION HISTORY:
! SJL 03.11.04: Initial version for partial remapping
!
!-----------------------------------------------------------------------
  integer :: i,j,k 
     real q_source(is:ie,js:je,nq)    ! numerical tracer source from surface
                                      ! in case fillz is not sufficient
     real te_2d(is:ie,js:je)
      real zsum0(is:ie,js:je)
      real zsum1(is:ie,js:je)
      real   q2(is:ie,km)
      real  dp2(is:ie,km)
      real  pe1(is:ie,km+1)
      real  pe2(is:ie,km+1)
      real  pk1(is:ie,km+1)
      real  pk2(is:ie,km+1)
      real  pn2(is:ie,km+1)
      real  pe0(is:ie+1,km+1)
      real  pe3(is:ie+1,km+1)
      real phis(is:ie,km+1)
      real   gz(is:ie)
! for nonhydrostatic option with hybrid_z coordinate
      real ze1(is:ie,km+1), ze2(is:ie,km+1), deng(is:ie,km)

      real rcp, rg, ak1, tmp, tpe, cv, rgama
      real bkh
      real dtmp
      real dlnp
      integer iq, n, kp, k_next
      logical te_map
      real k1k, kapag

      k1k   =  akap / (1.-akap)
      kapag = -akap / grav
       rg = akap * cp
       cv = cp - rg
      rgama = cv/cp
      rcp = 1./ cp
      ak1 = (akap + 1.) / akap

      if ( kord_tm < 0 ) then
           te_map = .false.
!          do j=js,je
!             do k=2,km+1
!                do i=is,ie
!                   peln(i,k,j) = log(pe(i,k,j))
!                enddo
!             enddo
!          enddo
          if ( remap_t ) then
! Transform virtual pt to virtual Temp
             do k=1,km
                do j=js,je
                   do i=is,ie
                      pt(i,j,k) = pt(i,j,k) * (pk(i,j,k+1)-pk(i,j,k)) /  &
                                 (rg*(peln(i,k+1,j)-peln(i,k,j)) )
                   enddo
                enddo
             enddo
          endif 
      else
           te_map = .true.
           call pkez(km, is, ie, js, je, pe, pk, akap, peln, pkz)
           call cubed_to_latlon(u, v, ua, va, dx, dy, rdxa, rdya, km, 1)
! Compute cp*T + KE
           do k=1,km
              do j=js,je
                 do i=is,ie
                    te(i,j,k) = 0.5*(ua(i,j,k)**2 + va(i,j,k)**2)  &
                                  +  pt(i,j,k)*pkz(i,j,k)
                 enddo
              enddo
           enddo
     endif

     if ( (.not.hydrostatic) .and. (.not.hybrid_z) ) then
           do k=1,km
              do j=js,je
                 do i=is,ie
                    delz(i,j,k) = -delz(i,j,k) / delp(i,j,k) ! ="specific volume"/grav
                 enddo
              enddo
           enddo
     endif

  do 1000 j=js,je+1

        do k=1,km+1
           do i=is,ie
              pe1(i,k) = pe(i,k,j)
           enddo
        enddo

        do i=is,ie
           pe2(i,   1) = ptop
           pe2(i,km+1) = pe(i,km+1,j)
        enddo

  if ( j < (je+1) )  then 
! update ps
        do i=is,ie
            ps(i,j) = pe1(i,km+1)
        enddo

   if ( hybrid_z ) then
!--------------------------
! hybrid z_p coordinate
!--------------------------

        do i=is,ie
           ze1(i,km+1) = ze0(i,j,km+1)
        enddo

        do k=km,1,-1
           do i=is,ie
              ze1(i,k) = ze1(i,k+1) - delz(i,j,k)   ! current height
           enddo
        enddo

! Copy ztop; the top layer must be thick enough to prevent numerical problems.
!
        do i=is,ie
           ze2(i,  1) = ze1(i,1)
           ze0(i,j,1) = ze1(i,1)      ! Note: ze0 (top) updated
        enddo

        do k=2,km+1
           do i=is,ie
              ze2(i,k) = ze0(i,j,k)   ! specified height
           enddo
        enddo
!

        do k=1,km
           do i=is,ie
              deng(i,k) = -delp(i,j,k)/delz(i,j,k)  ! density * grav
           enddo
        enddo

        call remap_z(km, ze1, deng, km, ze2, deng, is, ie, abs(kord_tm))
!-------------
! Update delz
!-------------
        do k=1,km
           do i=is,ie
              delz(i,j,k) = ze2(i,k+1) - ze2(i,k)
           enddo
        enddo

!------------
! update delp
!------------
        do k=1,km-1
           do i=is,ie
               dp2(i,k  ) = -delz(i,j,k)*deng(i,k)
               pe2(i,k+1) =   pe2(i,k) +  dp2(i,k)
           enddo
        enddo

        do i=is,ie
           dp2(i,km) = pe2(i,km+1) - pe2(i,km)  ! to reduce rounding error
        enddo
   else
!
! Hybrid sigma-P coordinate:
!
        do k=2,ks+1
           do i=is,ie
              pe2(i,k) = ak(k)
           enddo
        enddo
        do k=ks+2,km
           do i=is,ie
              pe2(i,k) = ak(k) + bk(k)*pe(i,km+1,j)
           enddo
        enddo

        do k=1,km
           do i=is,ie
              dp2(i,k) = pe2(i,k+1) - pe2(i,k)
           enddo
        enddo
   endif

!------------
! update delp
!------------
      do k=1,km
         do i=is,ie
            delp(i,j,k) = dp2(i,k)
         enddo
      enddo

!----------------
! Map constituents
!----------------
       if( nq /= 0 ) then
!------------------------------------------------------------------
! Do remapping one tracer at a time; seems to be faster on the SGI
! It requires less memory than mapn_ppm
!------------------------------------------------------------------
          do iq=1,nq
             call map1_q2(km, pe1, q(isd,jsd,1,iq),     &
                          km, pe2, q2, dp2,             &
                          is, ie, 0, kord_tr, j, isd, ied, jsd, jed)
!           if (fill) call fillz(ie-is+1, km, 1, q2, dp2, q_source(is,j,iq))
            if (fill) call fillz(ie-is+1, km, 1, q2, dp2)
            do k=1,km
               do i=is,ie
                  q(i,j,k,iq) = q2(i,k)
               enddo
            enddo
          enddo
       endif

!------------------
! Compute p**cappa
!------------------
   do k=1,km+1
      do i=is,ie
         pk1(i,k) = pk(i,j,k)
      enddo
   enddo

   do i=is,ie
      pn2(i,   1) = peln(i,   1,j)
      pn2(i,km+1) = peln(i,km+1,j)
      pk2(i,   1) = pk1(i,   1)
      pk2(i,km+1) = pk1(i,km+1)
   enddo

   do k=2,km
      do i=is,ie
!        pk2(i,k) = pe2(i,k) ** akap
         pn2(i,k) = log(pe2(i,k))
         pk2(i,k) = exp(akap*pn2(i,k))
      enddo
   enddo

   if ( te_map ) then
!---------------------
! Compute Total Energy
!---------------------
        do i=is,ie
           phis(i,km+1) = hs(i,j)
        enddo
        do k=km,1,-1
           do i=is,ie
              phis(i,k) = phis(i,k+1) + pt(i,j,k)*(pk1(i,k+1)-pk1(i,k))
           enddo
        enddo
        do k=1,km+1
           do i=is,ie
              phis(i,k) = phis(i,k) * pe1(i,k)
           enddo
        enddo
        do k=1,km
           do i=is,ie
              te(i,j,k) = te(i,j,k)+(phis(i,k+1)-phis(i,k))/(pe1(i,k+1)-pe1(i,k))
           enddo
        enddo
!----------------
! Map Total Energy
!----------------
        call map1_ppm (km,   pe1,  te,       &
                       km,   pe2,  te,       &
                       is, ie, j, is, ie, js, je, 1, kord_tm)
   else
     if ( remap_t ) then
!----------------
! Map t using logp
!----------------
       call map1_ppm (km,  peln(is,1,j),  pt,    &
                      km,  pn2,           pt,    &
                      is, ie, j, isd, ied, jsd, jed, 1, abs(kord_tm))
     else
!----------------
! Map pt using pk
!----------------
       call map1_ppm (km,  pk1,  pt,           &
                      km,  pk2,  pt,           &
                      is, ie, j, isd, ied, jsd, jed, 1, abs(kord_tm))
     endif
   endif

   if ( .not. hydrostatic ) then
! Remap vertical wind:
        call map1_ppm (km,   pe1,  w,       &
                       km,   pe2,  w,       &
                       is, ie, j, isd, ied, jsd, jed, -1, kord_mt)
     if ( .not. hybrid_z ) then
! Remap delz for hybrid sigma-p coordinate
        call map1_ppm (km,   pe1, delz,    &
                       km,   pe2, delz,    &
                       is, ie, j, is,  ie,  js,  je,  1, abs(kord_tm))
        do k=1,km
           do i=is,ie
              delz(i,j,k) = -delz(i,j,k)*dp2(i,k)
           enddo
        enddo
     endif
   endif

!----------
! Update pk
!----------
   do k=2,km
      do i=is,ie
         pk(i,j,k) = pk2(i,k)
      enddo
   enddo

! Copy omega field to pe3
   do i=is,ie
      pe3(i,1) = 0.
   enddo
   do k=2,km+1
      do i=is,ie
         pe3(i,k) = omga(i,j,k-1)
      enddo
   enddo

   do k=1,km+1
      do i=is,ie
          pe0(i,k)   = peln(i,k,j)
         peln(i,k,j) =  pn2(i,k)
      enddo
   enddo

!------------
! Compute pkz
!------------
   if ( hydrostatic ) then
      do k=1,km
         do i=is,ie
            pkz(i,j,k) = (pk2(i,k+1)-pk2(i,k))/(akap*(peln(i,k+1,j)-peln(i,k,j)))
         enddo
      enddo
   else
      if ( ktop>1 ) then
         do k=1,ktop-1
         do i=is,ie
            pkz(i,j,k) = (pk2(i,k+1)-pk2(i,k))/(akap*(peln(i,k+1,j)-peln(i,k,j)))
         enddo
         enddo
      endif
      do k=ktop,km
         do i=is,ie
! Note: pt at this stage is cp*Theta_v
!           pkz(i,j,k) = ( kapag*delp(i,j,k)*pt(i,j,k) /            &
!                         (delz(i,j,k)*(1.+r_vir*q(i,j,k,sphum))) )**k1k
            pkz(i,j,k) = exp( k1k*log(kapag*delp(i,j,k)/delz(i,j,k)*pt(i,j,k)) )
         enddo
      enddo
   endif

! Interpolate omega/pe3 (defined at pe0) to remapped cell center (dp2)
   do k=1,km
      do i=is,ie
         dp2(i,k) = 0.5*(peln(i,k,j) + peln(i,k+1,j))
      enddo
   enddo

   do i=is,ie
       k_next = 1
       do n=1,km
          kp = k_next
          do k=kp,km
             if( dp2(i,n) <= pe0(i,k+1) .and. dp2(i,n) >= pe0(i,k) ) then
                 omga(i,j,n) = pe3(i,k)  +  (pe3(i,k+1) - pe3(i,k)) *    &
                       (dp2(i,n)-pe0(i,k)) / (pe0(i,k+1)-pe0(i,k) )
                 k_next = k
                 exit
             endif
          enddo
       enddo
   enddo

  endif !(j < je+1)

 if ( .not.hybrid_z ) then
      do i=is,ie+1
         pe0(i,1) = pe(i,1,j)
      enddo
!------
! map u
!------
      do k=2,km+1
         do i=is,ie
            pe0(i,k) = 0.5*(pe(i,k,j-1)+pe1(i,k))
         enddo
      enddo


      do k=1,ks+1
         do i=is,ie+1
            pe3(i,k) = ak(k)
         enddo
      enddo

      do k=ks+2,km+1
         bkh = 0.5*bk(k)
         do i=is,ie
            pe3(i,k) = ak(k) + bkh*(pe(i,km+1,j-1)+pe1(i,km+1))
         enddo
      enddo

      call map1_ppm( km, pe0(is:ie,:),   u,       &
                     km, pe3(is:ie,:),   u,       &
                     is, ie, j, isd, ied, jsd, jed+1, -1, kord_mt)
      if (present(mfy)) then
         call map1_ppm( km, pe0(is:ie,:), mfy,       &
                        km, pe3(is:ie,:), mfy,       &
                        is, ie, j, is, ie, js, je+1, -1, kord_mt)
      endif

   if (j < je+1) then
!------
! map v
!------
       do k=2,km+1
          do i=is,ie+1
             pe0(i ,k) = 0.5*(pe(i-1,k,j)+pe(i,k,j))
          enddo
       enddo
       do k=ks+2,km+1
          bkh = 0.5*bk(k)
          do i=is,ie+1
             pe3(i,k) = ak(k) + bkh*(pe(i-1,km+1,j)+pe(i,km+1,j))
          enddo
       enddo

       call map1_ppm (km, pe0,  v,              &
                      km, pe3,  v, is, ie+1,    &
                      j, isd, ied+1, jsd, jed, -1, kord_mt)
       if (present(mfx)) then
          call map1_ppm (km, pe0, mfx,              &
                         km, pe3, mfx, is, ie+1,    &
                         j, is, ie+1, js, je, -1, kord_mt)
       endif
   endif ! (j < je+1)
 endif    ! end hybrid_z check
     do k=1,km
        do i=is,ie
           ua(i,j,k) = pe2(i,k+1)
        enddo
     enddo

1000  continue

if ( hybrid_z ) then   !------- Hybrid_z section ---------------
     call mpp_update_domains(ua , domain,  whalo=1, ehalo=1,     &
                             shalo=1, nhalo=1, complete=.true.)
! u-wind
   do j=js,je+1
      do i=is,ie
         pe1(i,1) = ptop
         pe2(i,1) = ptop
      enddo
      do k=2,km+1
         do i=is,ie
            pe1(i,k) = 0.5*(pe(i,k,  j-1) + pe(i,k,j  ))
            pe2(i,k) = 0.5*(ua(i,j-1,k-1) + ua(i,j,k-1))
         enddo
      enddo

      call map1_ppm( km, pe1,   u,       &
                     km, pe2,   u,       &
                     is, ie, j, isd, ied, jsd, jed+1, -1, kord_mt)
      if (present(mfy)) then
         call map1_ppm( km, pe1, mfy,       &
                        km, pe2, mfy,       &
                        is, ie, j, is, ie, js, je+1, -1, kord_mt)
      endif
   enddo

! v-wind
   do j=js,je
      do i=is,ie+1
         pe0(i,1) = ptop
         pe3(i,1) = ptop
      enddo

      do k=2,km+1
         do i=is,ie+1
            pe0(i,k) = 0.5*(pe(i-1,k,j  ) + pe(i,k,j  ))
            pe3(i,k) = 0.5*(ua(i-1,j,k-1) + ua(i,j,k-1))
         enddo
      enddo

      call map1_ppm (km, pe0,  v,              &
                     km, pe3,  v, is, ie+1,    &
                     j, isd, ied+1, jsd, jed, -1, kord_mt)
      if (present(mfx)) then
         call map1_ppm (km, pe0, mfx,              &
                        km, pe3, mfx, is, ie+1,    &
                        j, is, ie+1, js, je, -1, kord_mt)
      endif
   enddo
endif         !------------- Hybrid_z section ----------------------

     do k=2,km
        do j=js,je
           do i=is,ie
              pe(i,k,j) = ua(i,j,k-1)
           enddo
        enddo
     enddo

  call cubed_to_latlon(u,  v, ua, va, dx, dy, rdxa, rdya, km, 1)

  if( consv > 0. ) then

    if ( te_map ) then
      do j=js,je
          do i=is,ie
             te_2d(i,j) = te(i,j,1)*delp(i,j,1)
          enddo
          do k=2,km
             do i=is,ie
                te_2d(i,j) = te_2d(i,j) + te(i,j,k)*delp(i,j,k)
             enddo
          enddo
      enddo
    else
      do j=js,je
        if ( remap_t ) then
         do i=is,ie
            gz(i) = hs(i,j)
            do k=1,km
               gz(i) = gz(i) + rg*pt(i,j,k)*(peln(i,k+1,j)-peln(i,k,j))
            enddo
         enddo
         do i=is,ie
            te_2d(i,j) = pe(i,km+1,j)*hs(i,j) - pe(i,1,j)*gz(i)
         enddo

         do k=1,km
            do i=is,ie
               te_2d(i,j) = te_2d(i,j) + delp(i,j,k)*(cp*pt(i,j,k) +   &  
                                      0.5*(ua(i,j,k)**2+va(i,j,k)**2))
            enddo
         enddo
        else
         if ( hydrostatic ) then
            do i=is,ie
               gz(i) = hs(i,j)
               do k=1,km
                  gz(i) = gz(i) + pt(i,j,k)*(pk(i,j,k+1)-pk(i,j,k))
               enddo
            enddo

            do i=is,ie
               te_2d(i,j) = pe(i,km+1,j)*hs(i,j) - pe(i,1,j)*gz(i)
            enddo
            do k=1,km
               do i=is,ie
                  te_2d(i,j) = te_2d(i,j) + delp(i,j,k)*(pt(i,j,k)*pkz(i,j,k) +   &  
                               0.5*(ua(i,j,k)**2+va(i,j,k)**2))
               enddo
            enddo
         else
!-----------------
! Non-hydrostatic:
!-----------------
           do i=is,ie
              phis(i,km+1) = hs(i,j)
              do k=km,1,-1
                 phis(i,k) = phis(i,k+1) - grav*delz(i,j,k)
              enddo
           enddo
           do i=is,ie
              te_2d(i,j) = 0.
           enddo
           do k=1,km
              do i=is,ie
                 te_2d(i,j) = te_2d(i,j) + delp(i,j,k)*( rgama*pt(i,j,k)*pkz(i,j,k) +   &
                              0.5*(phis(i,k)+phis(i,k+1)+ua(i,j,k)**2+va(i,j,k)**2+w(i,j,k)**2) )
              enddo
           enddo
         endif

       endif
      enddo
    endif

      do j=js,je
         do i=is,ie
            zsum1(i,j) = pkz(i,j,1)*delp(i,j,1)
         enddo
         do k=2,km
            do i=is,ie
               zsum1(i,j) = zsum1(i,j) + pkz(i,j,k)*delp(i,j,k)
            enddo
         enddo

         do i=is,ie
            zsum0(i,j) = ptop*(pk(i,j,1)-pk(i,j,km+1)) + zsum1(i,j)
            te_2d(i,j) = te0_2d(i,j) - te_2d(i,j)
         enddo
      enddo

         tpe = consv*g_sum(te_2d, is, ie, js, je, ng, area, 0)
      E_Flux = tpe / (grav*mdt*4.*pi*radius**2)    ! unit: W/m**2

      if ( hydrostatic ) then
           dtmp = tpe / (cp*g_sum(zsum0,  is, ie, js, je, ng, area, 0))
      else
           dtmp = tpe / (cv*g_sum(zsum1,  is, ie, js, je, ng, area, 0))
      endif
!-------------------------------------------------------------------------------
! One may use this quick fix to ensure reproducibility at the expense of a lower
! floating precision; this is fine for the TE correction
!-------------------------------------------------------------------------------
      if ( reproduce_sum ) dtmp = real(dtmp, 4) ! convert to 4-byte real
  else
      dtmp   = 0.
      E_Flux = 0.
  endif        ! end consv check

  if ( te_map ) then
      do j=js,je
         do i=is,ie
            gz(i) = hs(i,j)
         enddo
         do k=km,1,-1
            do i=is,ie
               tpe = te(i,j,k) - 0.5*(ua(i,j,k)**2 + va(i,j,k)**2) - gz(i)
               dlnp = rg*(peln(i,k+1,j) - peln(i,k,j))
               tmp = tpe / ((cp - pe(i,k,j)*dlnp/delp(i,j,k))*(1.+r_vir*q(i,j,k,sphum)) )
               pt(i,j,k) =  tmp + dtmp*pkz(i,j,k) / (1.+r_vir*q(i,j,k,sphum))
               gz(i) = gz(i) + dlnp*tmp*(1.+r_vir*q(i,j,k,sphum))
            enddo
         enddo           ! end k-loop
      enddo
  else
    if ( remap_t ) then
      do k=1,km
         do j=js,je
            do i=is,ie
               pt(i,j,k) = (pt(i,j,k) + dtmp*pkz(i,j,k))/(1.+r_vir*q(i,j,k,sphum))
            enddo
         enddo   
      enddo
    else
      do k=1,km
         do j=js,je
            do i=is,ie
               pt(i,j,k) = (rcp*pt(i,j,k) + dtmp)*pkz(i,j,k)/(1.+r_vir*q(i,j,k,sphum))
            enddo
         enddo   
      enddo
    endif
  endif

 end subroutine Lagrangian_to_Eulerian


 subroutine compute_total_energy(is, ie, js, je, isd, ied, jsd, jed, km,  &
                                 u, v, w, delz, pt, delp, q, pe, peln, hs, &
                                 r_vir,  cp, rg, hlv, te_2d, ua, va, teq, &
                                 moist_phys, sphum, hydrostatic, id_te)
!------------------------------------------------------
! Compute vertically integrated total energy per column
!------------------------------------------------------
! !INPUT PARAMETERS:
   integer,  intent(in):: km, is, ie, js, je, isd, ied, jsd, jed, id_te
   integer,  intent(in):: sphum
   real, intent(in), dimension(isd:ied,jsd:jed,km):: pt, delp
   real, intent(in), dimension(isd:ied,jsd:jed,km,sphum):: q
   real, intent(inout)::  u(isd:ied,  jsd:jed+1,km)
   real, intent(inout)::  v(isd:ied+1,jsd:jed,  km)
   real, intent(in)::  w(isd:ied,jsd:jed,km)   ! vertical velocity (m/s)
   real, intent(in):: delz(is:ie,js:je,km)
   real, intent(in):: hs(isd:ied,jsd:jed)  ! surface geopotential
   real, intent(in)::   pe(is-1:ie+1,km+1,js-1:je+1) ! pressure at layer edges
   real, intent(in):: peln(is:ie,km+1,js:je)  ! log(pe)
   real, intent(in):: cp, rg, r_vir, hlv
   logical, intent(in):: moist_phys, hydrostatic
! Output:
   real, intent(out), dimension(isd:ied,jsd:jed,km):: ua, va
   real, intent(out):: te_2d(is:ie,js:je)   ! vertically integrated TE
   real, intent(out)::   teq(is:ie,js:je)   ! Moist TE
! Local
   real  gztop(is:ie)
   real  phiz(is:ie,km+1)
   real cv
   integer i, j, k

   cv = cp - rg

!----------------------
! Output lat-lon winds:
!----------------------
  call cubed_to_latlon(u, v, ua, va, dx, dy, rdxa, rdya, km)

  do j=js,je

     if ( hydrostatic ) then

     do i=is,ie
        gztop(i) = hs(i,j)
        do k=1,km
           gztop(i) = gztop(i) + (peln(i,k+1,j)-peln(i,k,j)) *   &
                      rg*pt(i,j,k)*(1.+r_vir*q(i,j,k,sphum))
        enddo
     enddo
     do i=is,ie
        te_2d(i,j) = pe(i,km+1,j)*hs(i,j) - pe(i,1,j)*gztop(i)
     enddo

     do k=1,km
        do i=is,ie
           te_2d(i,j) = te_2d(i,j) + delp(i,j,k)*(0.5*(ua(i,j,k)**2+va(i,j,k)**2)  &
                      + cp*pt(i,j,k)*(1.+r_vir*q(i,j,k,sphum)))
        enddo
     enddo
     else
! Non-hydrostatic:
     do i=is,ie
        phiz(i,km+1) = hs(i,j)
        do k=km,1,-1
           phiz(i,k) = phiz(i,k+1) - grav*delz(i,j,k)
        enddo
     enddo
     do i=is,ie
        te_2d(i,j) = 0.
     enddo
     do k=1,km
        do i=is,ie
           te_2d(i,j) = te_2d(i,j) + delp(i,j,k)*( cv*pt(i,j,k)*(1.+r_vir*q(i,j,k,sphum)) +  &
                        0.5*(phiz(i,k)+phiz(i,k+1)+ua(i,j,k)**2+va(i,j,k)**2+w(i,j,k)**2) )
        enddo
     enddo
     endif
  enddo

!-------------------------------------
! Doganostics computation for moist TE
!-------------------------------------
  if( id_te>0 ) then
      do j=js,je
         do i=is,ie
            teq(i,j) = te_2d(i,j)
         enddo
      enddo
      if ( moist_phys ) then
           do k=1,km
              do j=js,je
                 do i=is,ie
                    teq(i,j) = teq(i,j) + hlv*q(i,j,k,sphum)*delp(i,j,k)
                 enddo
              enddo
           enddo
      endif
!     do j=js,je
!        do i=is,ie
!           teq(i,j) = teq(i,j) / (pe(i,km,j) - pe(i,1,j))
!        enddo
!     enddo
   endif

  end subroutine compute_total_energy


  subroutine pkez(km, ifirst, ilast, jfirst, jlast, &
                  pe, pk, akap, peln, pkz)

! !INPUT PARAMETERS:
   integer, intent(in):: km
   integer, intent(in):: ifirst, ilast        ! Latitude strip
   integer, intent(in):: jfirst, jlast        ! Latitude strip
   real, intent(in):: akap
   real, intent(in):: pe(ifirst-1:ilast+1,km+1,jfirst-1:jlast+1)
   real, intent(in):: pk(ifirst:ilast,jfirst:jlast,km+1)
! !OUTPUT
   real, intent(out):: pkz(ifirst:ilast,jfirst:jlast,km)
   real, intent(out):: peln(ifirst:ilast, km+1, jfirst:jlast)   ! log (pe)
! Local
   real pk2(ifirst:ilast, km+1)
   real pek
   real lnp
   real ak1
   integer i, j, k

   ak1 = (akap + 1.) / akap

   do j=jfirst, jlast
        pek = pk(ifirst,j,1)
        do i=ifirst, ilast
           pk2(i,1) = pek
        enddo

        do k=2,km+1
           do i=ifirst, ilast
!             peln(i,k,j) =  log(pe(i,k,j))
              pk2(i,k) =  pk(i,j,k)
           enddo
        enddo

!---- GFDL modification
       if( ptop < ptop_min ) then
           do i=ifirst, ilast
               peln(i,1,j) = peln(i,2,j) - ak1
           enddo
       else
           lnp = log( ptop )
           do i=ifirst, ilast
              peln(i,1,j) = lnp
           enddo
       endif
!---- GFDL modification

       do k=1,km
          do i=ifirst, ilast
             pkz(i,j,k) = (pk2(i,k+1) - pk2(i,k) )  /  &
                          (akap*(peln(i,k+1,j) - peln(i,k,j)) )
          enddo
       enddo
    enddo

 end subroutine pkez



 subroutine remap_z(km, pe1, q1, kn, pe2, q2, i1, i2, kord)

! !INPUT PARAMETERS:
      integer, intent(in) :: i1                ! Starting longitude
      integer, intent(in) :: i2                ! Finishing longitude
      integer, intent(in) :: kord              ! Method order
      integer, intent(in) :: km                ! Original vertical dimension
      integer, intent(in) :: kn                ! Target vertical dimension

      real, intent(in) ::  pe1(i1:i2,km+1)     ! height at layer edges 
                                               ! (from model top to bottom surface)
      real, intent(in) ::  pe2(i1:i2,kn+1)     ! hieght at layer edges 
                                               ! (from model top to bottom surface)
      real, intent(in) ::  q1(i1:i2,km)        ! Field input

! !INPUT/OUTPUT PARAMETERS:
      real, intent(inout)::  q2(i1:i2,kn)      ! Field output

! !LOCAL VARIABLES:
      real  dp1(  i1:i2,km)
      real   q4(4,i1:i2,km)
      real   pl, pr, qsum, delp, esl
      integer i, k, l, m, k0
#if !defined(TAF_DOES_NOT_LIKE)
      logical :: l_123,m_123,l_555,l_loop
      real, parameter::  r3 = 1./3., r23 = 2./3., r12 = 1./12.
#endif

      do k=1,km
         do i=i1,i2
             dp1(i,k) = pe1(i,k+1) - pe1(i,k)      ! negative
            q4(1,i,k) = q1(i,k)
         enddo
      enddo

! Compute vertical subgrid distribution
   if ( kord >7 ) then
        call  cs_profile( q4, dp1, km, i1, i2, 1 )
   else
        call ppm_profile( q4, dp1, km, i1, i2, 1, kord )
   endif

#if defined(TAF_DOES_NOT_LIKE)
! Mapping
      do 1000 i=i1,i2
         k0 = 1
      do 555 k=1,kn
      do 100 l=k0,km
! locate the top edge: pe2(i,k)
      if(pe2(i,k) <= pe1(i,l) .and. pe2(i,k) >= pe1(i,l+1)) then
         pl = (pe2(i,k)-pe1(i,l)) / dp1(i,l)
         if(pe2(i,k+1) >= pe1(i,l+1)) then
! entire new grid is within the original grid
            pr = (pe2(i,k+1)-pe1(i,l)) / dp1(i,l)
            q2(i,k) = q4(2,i,l) + 0.5*(q4(4,i,l)+q4(3,i,l)-q4(2,i,l))  &
                       *(pr+pl)-q4(4,i,l)*r3*(pr*(pr+pl)+pl**2)
               k0 = l
               goto 555
          else
! Fractional area...
            qsum = (pe1(i,l+1)-pe2(i,k))*(q4(2,i,l)+0.5*(q4(4,i,l)+   &
                    q4(3,i,l)-q4(2,i,l))*(1.+pl)-q4(4,i,l)*           &
                     (r3*(1.+pl*(1.+pl))))
              do m=l+1,km
! locate the bottom edge: pe2(i,k+1)
                 if(pe2(i,k+1) < pe1(i,m+1) ) then
! Whole layer..
                    qsum = qsum + dp1(i,m)*q4(1,i,m)
                 else
                    delp = pe2(i,k+1)-pe1(i,m)
                    esl = delp / dp1(i,m)
                    qsum = qsum + delp*(q4(2,i,m)+0.5*esl*               &
                         (q4(3,i,m)-q4(2,i,m)+q4(4,i,m)*(1.-r23*esl)))
                    k0 = m
                 goto 123
                 endif
              enddo
              goto 123
           endif
      endif
100   continue
123   q2(i,k) = qsum / ( pe2(i,k+1) - pe2(i,k) )
555   continue
1000  continue
#else
! Mapping
      do i=i1,i2
         call mapz(pe1(i,:), pe2(i,:), dp1(i,:), q4(:,i,:), q2(i,:), km, kn, 0)
      enddo
#endif
 end subroutine remap_z


 subroutine map1_ppm( km,   pe1,    q1,                 &
                      kn,   pe2,    q2,   i1, i2,       &
                      j,    ibeg, iend, jbeg, jend, iv,  kord)
 integer, intent(in) :: i1                ! Starting longitude
 integer, intent(in) :: i2                ! Finishing longitude
 integer, intent(in) :: iv                ! Mode: 0 ==  constituents  1 == ???
                                          !       2 = potential temp
 integer, intent(in) :: kord              ! Method order
 integer, intent(in) :: j                 ! Current latitude
 integer, intent(in) :: ibeg, iend, jbeg, jend
 integer, intent(in) :: km                ! Original vertical dimension
 integer, intent(in) :: kn                ! Target vertical dimension
 real, intent(in) ::  pe1(i1:i2,km+1)  ! pressure at layer edges 
                                       ! (from model top to bottom surface)
                                       ! in the original vertical coordinate
 real, intent(in) ::  pe2(i1:i2,kn+1)  ! pressure at layer edges 
                                       ! (from model top to bottom surface)
                                       ! in the new vertical coordinate
 real, intent(in) ::    q1(ibeg:iend,jbeg:jend,km) ! Field input
! !INPUT/OUTPUT PARAMETERS:
 real, intent(inout)::  q2(ibeg:iend,jbeg:jend,kn) ! Field output

! !DESCRIPTION:
! IV = 0: constituents
! pe1: pressure at layer edges (from model top to bottom surface)
!      in the original vertical coordinate
! pe2: pressure at layer edges (from model top to bottom surface)
!      in the new vertical coordinate
! !LOCAL VARIABLES:
   real    dp1(i1:i2,km)
   real   q4(4,i1:i2,km)
   real    pl, pr, qsum, dp, delp, esl
   integer i, k, l, m, k0

#if !defined(TAF_DOES_NOT_LIKE)
      logical :: l_123,m_123,l_555,l_loop
      real, parameter::  r3 = 1./3., r23 = 2./3., r12 = 1./12.
#endif

   do k=1,km
      do i=i1,i2
         dp1(i,k) = pe1(i,k+1) - pe1(i,k)
         q4(1,i,k) = q1(i,j,k)
      enddo
   enddo

! Compute vertical subgrid distribution
   if ( kord >7 ) then
        call  cs_profile( q4, dp1, km, i1, i2, iv )
   else
        call ppm_profile( q4, dp1, km, i1, i2, iv, kord )
   endif

#if defined(TAF_DOES_NOT_LIKE)
  do i=i1,i2
     k0 = 1
     do 555 k=1,kn
      do l=k0,km
! locate the top edge: pe2(i,k)
      if( pe2(i,k) >= pe1(i,l) .and. pe2(i,k) <= pe1(i,l+1) ) then
         pl = (pe2(i,k)-pe1(i,l)) / dp1(i,l)
         if( pe2(i,k+1) <= pe1(i,l+1) ) then
! entire new grid is within the original grid
            pr = (pe2(i,k+1)-pe1(i,l)) / dp1(i,l)
            q2(i,j,k) = q4(2,i,l) + 0.5*(q4(4,i,l)+q4(3,i,l)-q4(2,i,l))  &
                       *(pr+pl)-q4(4,i,l)*r3*(pr*(pr+pl)+pl**2)
               k0 = l
               goto 555
         else
! Fractional area...
            qsum = (pe1(i,l+1)-pe2(i,k))*(q4(2,i,l)+0.5*(q4(4,i,l)+   &
                    q4(3,i,l)-q4(2,i,l))*(1.+pl)-q4(4,i,l)*           &
                     (r3*(1.+pl*(1.+pl))))
              do m=l+1,km
! locate the bottom edge: pe2(i,k+1)
                 if( pe2(i,k+1) > pe1(i,m+1) ) then
! Whole layer
                     qsum = qsum + dp1(i,m)*q4(1,i,m)
                 else
                     dp = pe2(i,k+1)-pe1(i,m)
                     esl = dp / dp1(i,m)
                     qsum = qsum + dp*(q4(2,i,m)+0.5*esl*               &
                           (q4(3,i,m)-q4(2,i,m)+q4(4,i,m)*(1.-r23*esl)))
                     k0 = m
                     goto 123
                 endif
              enddo
              goto 123
         endif
      endif
      enddo
123   q2(i,j,k) = qsum / ( pe2(i,k+1) - pe2(i,k) )
555   continue
  enddo
#else
! Mapping
      do i=i1,i2
         call mapz(pe1(i,:), pe2(i,:), dp1(i,:), q4(:,i,:), q2(i,j,:), km, kn, 1)
      enddo
#endif
 end subroutine map1_ppm


 subroutine map1_q2(km,   pe1,   q1,            &
                    kn,   pe2,   q2,   dp2,     &
                    i1,   i2,    iv,   kord, j, &
                    ibeg, iend, jbeg, jend )


! !INPUT PARAMETERS:
      integer, intent(in) :: j
      integer, intent(in) :: i1, i2
      integer, intent(in) :: ibeg, iend, jbeg, jend
      integer, intent(in) :: iv                ! Mode: 0 ==  constituents 1 == ???
      integer, intent(in) :: kord
      integer, intent(in) :: km                ! Original vertical dimension
      integer, intent(in) :: kn                ! Target vertical dimension

      real, intent(in) ::  pe1(i1:i2,km+1)     ! pressure at layer edges 
                                               ! (from model top to bottom surface)
                                               ! in the original vertical coordinate
      real, intent(in) ::  pe2(i1:i2,kn+1)     ! pressure at layer edges 
                                               ! (from model top to bottom surface)
                                               ! in the new vertical coordinate
      real, intent(in) ::  q1(ibeg:iend,jbeg:jend,km) ! Field input
      real, intent(in) ::  dp2(i1:i2,kn)
! !INPUT/OUTPUT PARAMETERS:
      real, intent(inout):: q2(i1:i2,kn) ! Field output
! !LOCAL VARIABLES:
      real   dp1(i1:i2,km)
      real   q4(4,i1:i2,km)
      real   pl, pr, qsum, dp, delp, esl

      integer i, k, l, m, k0

#if !defined(TAF_DOES_NOT_LIKE)
      logical :: l_123,m_123,l_555,l_loop
      real, parameter::  r3 = 1./3., r23 = 2./3., r12 = 1./12.
#endif

      do k=1,km
         do i=i1,i2
             dp1(i,k) = pe1(i,k+1) - pe1(i,k)
            q4(1,i,k) = q1(i,j,k)
         enddo
      enddo

! Compute vertical subgrid distribution
   if ( kord >7 ) then
        call  cs_profile( q4, dp1, km, i1, i2, iv )
   else
        call ppm_profile( q4, dp1, km, i1, i2, iv, kord )
   endif

#if defined(TAF_DOES_NOT_LIKE)
! Mapping
      do 1000 i=i1,i2
         k0 = 1
      do 555 k=1,kn
      do 100 l=k0,km
! locate the top edge: pe2(i,k)
      if(pe2(i,k) >= pe1(i,l) .and. pe2(i,k) <= pe1(i,l+1)) then
         pl = (pe2(i,k)-pe1(i,l)) / dp1(i,l)
         if(pe2(i,k+1) <= pe1(i,l+1)) then
! entire new grid is within the original grid
            pr = (pe2(i,k+1)-pe1(i,l)) / dp1(i,l)
            q2(i,k) = q4(2,i,l) + 0.5*(q4(4,i,l)+q4(3,i,l)-q4(2,i,l))  &
                       *(pr+pl)-q4(4,i,l)*r3*(pr*(pr+pl)+pl**2)
               k0 = l
               goto 555
          else
! Fractional area...
            qsum = (pe1(i,l+1)-pe2(i,k))*(q4(2,i,l)+0.5*(q4(4,i,l)+   &
                    q4(3,i,l)-q4(2,i,l))*(1.+pl)-q4(4,i,l)*           &
                     (r3*(1.+pl*(1.+pl))))
              do m=l+1,km
! locate the bottom edge: pe2(i,k+1)
                 if(pe2(i,k+1) > pe1(i,m+1) ) then
                                                   ! Whole layer..
                    qsum = qsum + dp1(i,m)*q4(1,i,m)
                 else
                     dp = pe2(i,k+1)-pe1(i,m)
                    esl = dp / dp1(i,m)
                   qsum = qsum + dp*(q4(2,i,m)+0.5*esl*               &
                       (q4(3,i,m)-q4(2,i,m)+q4(4,i,m)*(1.-r23*esl)))
                   k0 = m
                   goto 123
                 endif
              enddo
              goto 123
          endif
      endif
100   continue
123   q2(i,k) = qsum / dp2(i,k)
555   continue
1000  continue
#else
! Mapping
      do i=i1,i2
         call mapz(pe1(i,:), pe2(i,:), dp1(i,:), q4(:,i,:), q2(i,:), km, kn, 1) 
      enddo
#endif

 end subroutine map1_q2



 subroutine remap_2d(km,   pe1,   q1,        &
                     kn,   pe2,   q2,        &
                     i1,   i2,    iv,   kord )
   integer, intent(in):: i1, i2
   integer, intent(in):: iv               ! Mode: 0 ==  constituents 1 ==others
   integer, intent(in):: kord
   integer, intent(in):: km               ! Original vertical dimension
   integer, intent(in):: kn               ! Target vertical dimension
   real, intent(in):: pe1(i1:i2,km+1)     ! pressure at layer edges 
                                          ! (from model top to bottom surface)
                                          ! in the original vertical coordinate
   real, intent(in):: pe2(i1:i2,kn+1)     ! pressure at layer edges 
                                          ! (from model top to bottom surface)
                                          ! in the new vertical coordinate
   real, intent(in) :: q1(i1:i2,km) ! Field input
   real, intent(out):: q2(i1:i2,kn) ! Field output
! !LOCAL VARIABLES:
   real   dp1(i1:i2,km)
   real   q4(4,i1:i2,km)
   real   pl, pr, qsum, dp, delp, esl
   integer i, k, l, m, k0

#if !defined(TAF_DOES_NOT_LIKE)
   real, parameter::  r3 = 1./3., r23 = 2./3., r12 = 1./12.
#endif

   do k=1,km
      do i=i1,i2
          dp1(i,k) = pe1(i,k+1) - pe1(i,k)
         q4(1,i,k) = q1(i,k)
      enddo
   enddo

! Compute vertical subgrid distribution
   if ( kord >7 ) then
        call  cs_profile( q4, dp1, km, i1, i2, iv )
   else
        call ppm_profile( q4, dp1, km, i1, i2, iv, kord )
   endif

#if defined(TAF_DOES_NOT_LIKE)
   do i=i1,i2
      k0 = 1
      do 555 k=1,kn
         if( pe2(i,k+1) <= pe1(i,1) ) then
! Entire grid above old ptop
             q2(i,k) = q4(2,i,1)
         elseif( pe2(i,k) < pe1(i,1) .and. pe2(i,k+1)>pe1(i,1) ) then
! Partially above old ptop:
             q2(i,k) = q1(i,1)
         else
           do l=k0,km
! locate the top edge: pe2(i,k)
           if( pe2(i,k) >= pe1(i,l) .and. pe2(i,k) <= pe1(i,l+1) ) then
               pl = (pe2(i,k)-pe1(i,l)) / dp1(i,l)
               if(pe2(i,k+1) <= pe1(i,l+1)) then
! entire new grid is within the original grid
                  pr = (pe2(i,k+1)-pe1(i,l)) / dp1(i,l)
                  q2(i,k) = q4(2,i,l) + 0.5*(q4(4,i,l)+q4(3,i,l)-q4(2,i,l))  &
                          *(pr+pl)-q4(4,i,l)*r3*(pr*(pr+pl)+pl**2)
                  k0 = l
                  goto 555
               else
! Fractional area...
                 qsum = (pe1(i,l+1)-pe2(i,k))*(q4(2,i,l)+0.5*(q4(4,i,l)+   &
                         q4(3,i,l)-q4(2,i,l))*(1.+pl)-q4(4,i,l)*           &
                        (r3*(1.+pl*(1.+pl))))
                 do m=l+1,km
! locate the bottom edge: pe2(i,k+1)
                    if(pe2(i,k+1) > pe1(i,m+1) ) then
                                                   ! Whole layer..
                       qsum = qsum + dp1(i,m)*q4(1,i,m)
                    else
                       dp = pe2(i,k+1)-pe1(i,m)
                      esl = dp / dp1(i,m)
                      qsum = qsum + dp*(q4(2,i,m)+0.5*esl*               &
                            (q4(3,i,m)-q4(2,i,m)+q4(4,i,m)*(1.-r23*esl)))
                      k0 = m
                      goto 123
                    endif
                 enddo
                 goto 123
               endif
           endif
           enddo
123        q2(i,k) = qsum / ( pe2(i,k+1) - pe2(i,k) )
         endif
555   continue
   enddo
#else
! Mapping
      do i=i1,i2
         k0 = 1
         do k=1,kn

         if( pe2(i,k+1) <= pe1(i,1) ) then
! Entire grid above old ptop
            q2(i,k) = q4(2,i,1)
         elseif( pe2(i,k) < pe1(i,1) .and. pe2(i,k+1)>pe1(i,1) ) then
! Partially above old ptop:
            q2(i,k) = q1(i,1)
         else
            do l=k0,km
! locate the top edge: pe2(i,k)
            if(pe2(i,k) >= pe1(i,l) .and. pe2(i,k) <= pe1(i,l+1)) then
               pl = (pe2(i,k)-pe1(i,l)) / dp1(i,l)
               if(pe2(i,k+1) <= pe1(i,l+1)) then
! entire new grid is within the original grid
                  pr = (pe2(i,k+1)-pe1(i,l)) / dp1(i,l)
                  q2(i,k) = q4(2,i,l) + 0.5*(q4(4,i,l)+q4(3,i,l)-q4(2,i,l))  &
                             *(pr+pl)-q4(4,i,l)*r3*(pr*(pr+pl)+pl**2)
                  k0 = l
                  exit
               else
! Fractional area...
                  qsum = (pe1(i,l+1)-pe2(i,k))*(q4(2,i,l)+0.5*(q4(4,i,l)+   &
                          q4(3,i,l)-q4(2,i,l))*(1.+pl)-q4(4,i,l)*           &
                           (r3*(1.+pl*(1.+pl))))
                  do m=l+1,km
! locate the bottom edge: pe2(i,k+1)
                     if( pe2(i,k+1) > pe1(i,m+1) ) then
! Whole layer
                         qsum = qsum + dp1(i,m)*q4(1,i,m)
                     else
                         dp = pe2(i,k+1)-pe1(i,m)
                         esl = dp / dp1(i,m)
                         qsum = qsum + dp*(q4(2,i,m)+0.5*esl*               &
                               (q4(3,i,m)-q4(2,i,m)+q4(4,i,m)*(1.-r23*esl)))
                         k0 = m
                         q2(i,k) = qsum / ( pe2(i,k+1) - pe2(i,k) )
                         exit
                     endif
                  enddo
                  q2(i,k) = qsum / ( pe2(i,k+1) - pe2(i,k) )
                  exit
               endif
            endif
            enddo
         endif
         enddo
      enddo
#endif

 end subroutine remap_2d

 subroutine mapz(pe1, pe2, dp1, q4, q2, km, kn, order)
   integer, intent(in):: km         ! Original vertical dimension
   integer, intent(in):: kn         ! Target vertical dimension
   real, intent(in)::  pe1(km+1)    ! pressure at layer edges 
   real, intent(in)::  pe2(kn+1)    ! pressure at layer edges 
   real, intent(in)::  dp1(km)
   real, intent(in)::  q4(4,km)
   real, intent(out):: q2(kn)       ! Field output
   integer, intent(in):: order
! !LOCAL VARIABLES:
   real   pl, pr, qsum, dp, delp, esl
   integer k, l, m, k0

   logical :: i_am_done

   real, parameter::  r3 = 1./3., r23 = 2./3., r12 = 1./12.

     if (order==1) then
! 1 to KM  (TOP to BOTTOM)
         k0 = 1
         do k=1,kn
            i_am_done = .false.
            do l=k0,km
! locate the top edge: pe2(k)
            if ( (pe2(k) >= pe1(l) .and. pe2(k) <= pe1(l+1)) .and. (.not. i_am_done) ) then
               pl = (pe2(k)-pe1(l)) / dp1(l)
               if(pe2(k+1) <= pe1(l+1)) then
! entire new grid is within the original grid
                  pr = (pe2(k+1)-pe1(l)) / dp1(l)
                  q2(k) = q4(2,l) + 0.5*(q4(4,l)+q4(3,l)-q4(2,l))  &
                             *(pr+pl)-q4(4,l)*r3*(pr*(pr+pl)+pl**2)
                  k0 = l
                  i_am_done = .true.
               else
! Fractional area...
                  qsum = (pe1(l+1)-pe2(k))*(q4(2,l)+0.5*(q4(4,l)+   &
                          q4(3,l)-q4(2,l))*(1.+pl)-q4(4,l)*           &
                           (r3*(1.+pl*(1.+pl))))
                  do m=l+1,km
                     if( .not. i_am_done) then
! locate the bottom edge: pe2(k+1)
                       if( pe2(k+1) > pe1(m+1) ) then
! Whole layer
                         qsum = qsum + dp1(m)*q4(1,m)
                       else
                         dp = pe2(k+1)-pe1(m)
                         esl = dp / dp1(m)
                         qsum = qsum + dp*(q4(2,m)+0.5*esl*               &
                               (q4(3,m)-q4(2,m)+q4(4,m)*(1.-r23*esl)))
                         k0 = m
                         q2(k) = qsum / ( pe2(k+1) - pe2(k) )
                         i_am_done = .true.
                       endif
                     endif
                  enddo
                  if (.not. i_am_done) q2(k) = qsum / ( pe2(k+1) - pe2(k) )
                  i_am_done = .true.
               endif
            endif
            enddo
         enddo
     else
! 1 to KM  (BOTTOM to TOP)
         k0 = 1
         do k=1,kn
            i_am_done = .false.
            do l=k0,km
! locate the top edge: pe2(k)
            if ( (pe2(k) <= pe1(l) .and. pe2(k) >= pe1(l+1)) .and. (.not. i_am_done) ) then
               pl = (pe2(k)-pe1(l)) / dp1(l)
               if(pe2(k+1) >= pe1(l+1)) then
! entire new grid is within the original grid
                  pr = (pe2(k+1)-pe1(l)) / dp1(l)
                  q2(k) = q4(2,l) + 0.5*(q4(4,l)+q4(3,l)-q4(2,l))  &
                             *(pr+pl)-q4(4,l)*r3*(pr*(pr+pl)+pl**2)
                  k0 = l
                  i_am_done = .true.
               else
! Fractional area...   
                  qsum = (pe1(l+1)-pe2(k))*(q4(2,l)+0.5*(q4(4,l)+   &
                          q4(3,l)-q4(2,l))*(1.+pl)-q4(4,l)*           &
                           (r3*(1.+pl*(1.+pl))))
                  do m=l+1,km
                     if( .not. i_am_done) then
! locate the bottom edge: pe2(k+1)
                       if( pe2(k+1) < pe1(m+1) ) then
! Whole layer        
                         qsum = qsum + dp1(m)*q4(1,m)
                       else
                         dp = pe2(k+1)-pe1(m)
                         esl = dp / dp1(m)
                         qsum = qsum + dp*(q4(2,m)+0.5*esl*               &
                               (q4(3,m)-q4(2,m)+q4(4,m)*(1.-r23*esl)))
                         k0 = m
                         q2(k) = qsum / ( pe2(k+1) - pe2(k) )
                         i_am_done = .true.
                       endif 
                     endif
                  enddo 
                  if (.not. i_am_done) q2(k) = qsum / ( pe2(k+1) - pe2(k) )
                  i_am_done = .true.
               endif
            endif
            enddo
         enddo
     endif
 end subroutine mapz

 subroutine cs_profile(a4, delp, km, i1, i2, iv)
! Optimized vertical profile reconstruction:
! Latest: Apr 2008 S.-J. Lin, NOAA/GFDL
 integer, intent(in):: i1, i2
 integer, intent(in):: km      ! vertical dimension
 integer, intent(in):: iv      ! iv =-1: winds
                               ! iv = 0: positive definite scalars
                               ! iv = 1: others
 real , intent(in)   :: delp(i1:i2,km)     ! layer pressure thickness
 real , intent(inout):: a4(4,i1:i2,km)  ! Interpolated values
!-----------------------------------------------------------------------
 real  gam(i1:i2,km)
 real    q(i1:i2,km+1)
 real   d4(i1:i2)
 real   bet, a_bot, grat, pmp, lac
 integer i, k, im

  do i=i1,i2
         grat = delp(i,2) / delp(i,1)   ! grid ratio
          bet = grat*(grat+0.5)
       q(i,1) = ((grat+grat)*(grat+1.)*a4(1,i,1)+a4(1,i,2)) / bet
     gam(i,1) = ( 1. + grat*(grat+1.5) ) / bet
  enddo

  do k=2,km
     do i=i1,i2
           d4(i) = delp(i,k-1) / delp(i,k)
             bet =   2.+ d4(i) + d4(i) - gam(i,k-1)
          q(i,k) = ( 3.*(a4(1,i,k-1)+d4(i)*a4(1,i,k)) - q(i,k-1) )/bet
        gam(i,k) = d4(i) / bet
     enddo
  enddo
 
  do i=i1,i2
         a_bot = 1. + d4(i)*(d4(i)+1.5)
     q(i,km+1) = (2.*d4(i)*(d4(i)+1.)*a4(1,i,km)+a4(1,i,km-1)-a_bot*q(i,km))  &
               / ( d4(i)*(d4(i)+0.5) - a_bot*gam(i,km) )
  enddo

  do k=km,1,-1
     do i=i1,i2
        q(i,k) = q(i,k) - gam(i,k)*q(i,k+1)
     enddo
  enddo

!------------------
! Apply constraints
!------------------
  im = i2 - i1 + 1

  do k=2,km
     do i=i1,i2
        gam(i,k) = a4(1,i,k) - a4(1,i,k-1)
     enddo
  enddo

! Apply large-scale constraints to ALL fields if not local max/min
! Top:

  do i=i1,i2
     if ( (q(i,2)-q(i,1))*(q(i,3)-q(i,2))>0. ) then
          q(i,2) = min( q(i,2), max(a4(1,i,1), a4(1,i,2)) )
          q(i,2) = max( q(i,2), min(a4(1,i,1), a4(1,i,2)) )
     elseif ( iv==0 ) then
          q(i,2) = max(0., q(i,2))
     endif
  enddo

! Interior:
  do k=3,km-1
     do i=i1,i2
        if ( gam(i,k-1)*gam(i,k+1) > 0. ) then
! Apply large-scale constraint to ALL fields if not local max/min
             q(i,k) = min( q(i,k), max(a4(1,i,k-1),a4(1,i,k)) )
             q(i,k) = max( q(i,k), min(a4(1,i,k-1),a4(1,i,k)) )
#ifdef TEST_MONO
        elseif ( iv==0 ) then
             q(i,k) = max(0., q(i,k))
#else
        else
          if ( gam(i,k-1)>0. ) then
! There exists a local max
               q(i,k) = max(q(i,k), min(a4(1,i,k-1),a4(1,i,k)))
          else
            if ( iv==0 ) then
                 q(i,k) = max(0., q(i,k))
            else
! There exists a local min
                 q(i,k) = min(q(i,k), max(a4(1,i,k-1),a4(1,i,k)))
            endif
          endif
#endif
        endif
     enddo
  enddo


! Bottom:
  do i=i1,i2
     if ( (q(i,km)-q(i,km-1))*(q(i,km+1)-q(i,km))>0. ) then
          q(i,km) = min( q(i,km), max(a4(1,i,km-1), a4(1,i,km)) )
          q(i,km) = max( q(i,km), min(a4(1,i,km-1), a4(1,i,km)) )
     elseif ( iv==0 ) then
          q(i,km) = max(0., q(i,km))
     endif
  enddo

  do k=1,km
     do i=i1,i2
        a4(2,i,k) = q(i,k  )
        a4(3,i,k) = q(i,k+1)
     enddo
  enddo

! Top & bot surfaces
  do i=i1,i2
     if ( a4(2,i, 1)*a4(1,i, 1) <= 0. ) a4(2,i, 1) = 0.
     if ( a4(1,i,km)*a4(3,i,km) <= 0. ) a4(3,i,km) = 0.
  enddo

!-----------------------------------------------------------
! f(s) = AL + s*[(AR-AL) + A6*(1-s)]         ( 0 <= s  <= 1 )
!-----------------------------------------------------------
! Top 2 and bottom 2 layers always use monotonic mapping
  do k=1,2
     do i=i1,i2
        a4(4,i,k) = 3.*(2.*a4(1,i,k) - (a4(2,i,k)+a4(3,i,k)))
     enddo
     call cs_limiters(im, a4(1,i1,k), 1)
  enddo

!-------------------------------------
! Huynh's 2nd constraint for interior:
!-------------------------------------
  do k=3,km-2
     do i=i1,i2
! Left  edges
!      a4(2,i,k) = a4(1,i,k) -   &
!                  sign(min(abs(a4(2,i,k)-a4(1,i,k)), abs(gam(i,k))),gam(i,k))
!----
             pmp = a4(1,i,k) - 2.*gam(i,k+1)
             lac = pmp + 1.5*gam(i,k+2)
       a4(2,i,k) = min(max(a4(2,i,k),  min(a4(1,i,k), pmp, lac)),   &
                                       max(a4(1,i,k), pmp, lac) )
! Right edges
!      a4(3,i,k) = a4(1,i,k) +  &
!                  sign(min(abs(a4(3,i,k)-a4(1,i,k)), abs(gam(i,k+1))),gam(i,k+1))
!----
             pmp = a4(1,i,k) + 2.*gam(i,k)
             lac = pmp - 1.5*gam(i,k-1)
       a4(3,i,k) = min(max(a4(3,i,k), min(a4(1,i,k), pmp, lac)),    &
                                      max(a4(1,i,k), pmp, lac) )
       a4(4,i,k) = 3.*(2.*a4(1,i,k) - (a4(2,i,k)+a4(3,i,k)))
     enddo
! Additional constraint to ensure positivity
     if ( iv==0 ) call cs_limiters(im, a4(1,i1,k), 0)
  enddo

  do k=km-1,km
     do i=i1,i2
        a4(4,i,k) = 3.*(2.*a4(1,i,k) - (a4(2,i,k)+a4(3,i,k)))
     enddo
     call cs_limiters(im, a4(1,i1,k), 1)
  enddo

 end subroutine cs_profile



 subroutine cs_limiters(im, a4, iv)
 integer, intent(in) :: im
 integer, intent(in) :: iv
 real , intent(inout) :: a4(4,im)   ! PPM array
! !LOCAL VARIABLES:
 real  da1, da2, a6da
 real  fmin
 integer i

#if !defined(TAF_DOES_NOT_LIKE)
 real, parameter::  r3 = 1./3., r23 = 2./3., r12 = 1./12.
#endif

 if ( iv==0 ) then
! Positive definite constraint
    do i=1,im
      if( abs(a4(3,i)-a4(2,i)) < -a4(4,i) ) then
         fmin = a4(1,i)+0.25*(a4(3,i)-a4(2,i))**2/a4(4,i)+a4(4,i)*r12
         if( fmin < 0. ) then
             if( a4(1,i)<a4(3,i) .and. a4(1,i)<a4(2,i) ) then
                 a4(3,i) = a4(1,i)
                 a4(2,i) = a4(1,i)
                a4(4,i) = 0.
             elseif( a4(3,i) > a4(2,i) ) then
                 a4(4,i) = 3.*(a4(2,i)-a4(1,i))
                 a4(3,i) = a4(2,i) - a4(4,i)
             else
                 a4(4,i) = 3.*(a4(3,i)-a4(1,i))
                 a4(2,i) = a4(3,i) - a4(4,i)
             endif
         endif
      endif
    enddo
 else
! Standard PPM constraint
    do i=1,im
      if( (a4(1,i)-a4(2,i))*(a4(1,i)-a4(3,i))>=0. ) then
         a4(2,i) = a4(1,i)
         a4(3,i) = a4(1,i)
         a4(4,i) = 0.
      else
         da1  = a4(3,i) - a4(2,i)
         da2  = da1**2
         a6da = a4(4,i)*da1
         if(a6da < -da2) then
            a4(4,i) = 3.*(a4(2,i)-a4(1,i))
            a4(3,i) = a4(2,i) - a4(4,i)
         elseif(a6da > da2) then
            a4(4,i) = 3.*(a4(3,i)-a4(1,i))
            a4(2,i) = a4(3,i) - a4(4,i)
         endif
      endif
    enddo
 endif
 end subroutine cs_limiters



 subroutine ppm_profile(a4, delp, km, i1, i2, iv, kord)

! !INPUT PARAMETERS:
 integer, intent(in):: iv      ! iv =-1: winds
                               ! iv = 0: positive definite scalars
                               ! iv = 1: others
 integer, intent(in):: i1      ! Starting longitude
 integer, intent(in):: i2      ! Finishing longitude
 integer, intent(in):: km      ! vertical dimension
 integer, intent(in):: kord    ! Order (or more accurately method no.):
                               ! 
 real , intent(in):: delp(i1:i2,km)     ! layer pressure thickness

! !INPUT/OUTPUT PARAMETERS:
 real , intent(inout):: a4(4,i1:i2,km)  ! Interpolated values

! DESCRIPTION:
!
!   Perform the piecewise parabolic reconstruction
! 
! !REVISION HISTORY: 
! S.-J. Lin   revised at GFDL 2007
!-----------------------------------------------------------------------
! local arrays:
      real    dc(i1:i2,km)
      real    h2(i1:i2,km)
      real  delq(i1:i2,km)
      real   df2(i1:i2,km)
      real    d4(i1:i2,km)

! local scalars:
      integer i, k, km1, lmt, it
      real  fac
      real  a1, a2, c1, c2, c3, d1, d2
      real  qm, dq, lac, qmp, pmp

      km1 = km - 1
       it = i2 - i1 + 1

      do k=2,km
         do i=i1,i2
            delq(i,k-1) =   a4(1,i,k) - a4(1,i,k-1)
              d4(i,k  ) = delp(i,k-1) + delp(i,k)
         enddo
      enddo

      do k=2,km1
         do i=i1,i2
                 c1  = (delp(i,k-1)+0.5*delp(i,k))/d4(i,k+1)
                 c2  = (delp(i,k+1)+0.5*delp(i,k))/d4(i,k)
            df2(i,k) = delp(i,k)*(c1*delq(i,k) + c2*delq(i,k-1)) /      &
                                    (d4(i,k)+delp(i,k+1))
            dc(i,k) = sign( min(abs(df2(i,k)),              &
                            max(a4(1,i,k-1),a4(1,i,k),a4(1,i,k+1))-a4(1,i,k),  &
                  a4(1,i,k)-min(a4(1,i,k-1),a4(1,i,k),a4(1,i,k+1))), df2(i,k) )
         enddo
      enddo

!-----------------------------------------------------------
! 4th order interpolation of the provisional cell edge value
!-----------------------------------------------------------

      do k=3,km1
         do i=i1,i2
            c1 = delq(i,k-1)*delp(i,k-1) / d4(i,k)
            a1 = d4(i,k-1) / (d4(i,k) + delp(i,k-1))
            a2 = d4(i,k+1) / (d4(i,k) + delp(i,k))
            a4(2,i,k) = a4(1,i,k-1) + c1 + 2./(d4(i,k-1)+d4(i,k+1)) *    &
                      ( delp(i,k)*(c1*(a1 - a2)+a2*dc(i,k-1)) -          &
                        delp(i,k-1)*a1*dc(i,k  ) )
         enddo
      enddo

      if(km>8 .and. kord>3) call steepz(i1, i2, km, a4, df2, dc, delq, delp, d4)

! Area preserving cubic with 2nd deriv. = 0 at the boundaries
! Top
      do i=i1,i2
         d1 = delp(i,1)
         d2 = delp(i,2)
         qm = (d2*a4(1,i,1)+d1*a4(1,i,2)) / (d1+d2)
         dq = 2.*(a4(1,i,2)-a4(1,i,1)) / (d1+d2)
         c1 = 4.*(a4(2,i,3)-qm-d2*dq) / ( d2*(2.*d2*d2+d1*(d2+3.*d1)) )
         c3 = dq - 0.5*c1*(d2*(5.*d1+d2)-3.*d1*d1)
         a4(2,i,2) = qm - 0.25*c1*d1*d2*(d2+3.*d1)
! Top edge:
!-------------------------------------------------------
         a4(2,i,1) = d1*(2.*c1*d1**2-c3) + a4(2,i,2)
!-------------------------------------------------------
!        a4(2,i,1) = (12./7.)*a4(1,i,1)-(13./14.)*a4(1,i,2)+(3./14.)*a4(1,i,3)
!-------------------------------------------------------
! No over- and undershoot condition
         a4(2,i,2) = max( a4(2,i,2), min(a4(1,i,1), a4(1,i,2)) )
         a4(2,i,2) = min( a4(2,i,2), max(a4(1,i,1), a4(1,i,2)) )
         dc(i,1) =  0.5*(a4(2,i,2) - a4(1,i,1))
      enddo

! Enforce monotonicity of the "slope" within the top layer

      if( iv==0 ) then
         do i=i1,i2
            a4(2,i,1) = max(0., a4(2,i,1))
            a4(2,i,2) = max(0., a4(2,i,2))
         enddo 
      elseif( iv==-1 ) then
         do i=i1,i2
            if ( a4(2,i,1)*a4(1,i,1) <= 0. ) then
                 a4(2,i,1) = 0.
            endif
         enddo
      endif


! Bottom
! Area preserving cubic with 2nd deriv. = 0 at the surface
      do i=i1,i2
         d1 = delp(i,km)
         d2 = delp(i,km1)
         qm = (d2*a4(1,i,km)+d1*a4(1,i,km1)) / (d1+d2)
         dq = 2.*(a4(1,i,km1)-a4(1,i,km)) / (d1+d2)
         c1 = (a4(2,i,km1)-qm-d2*dq) / (d2*(2.*d2*d2+d1*(d2+3.*d1)))
         c3 = dq - 2.0*c1*(d2*(5.*d1+d2)-3.*d1*d1)
         a4(2,i,km) = qm - c1*d1*d2*(d2+3.*d1)
! Bottom edge:
!-----------------------------------------------------
         a4(3,i,km) = d1*(8.*c1*d1**2-c3) + a4(2,i,km)
!        dc(i,km) = 0.5*(a4(3,i,km) - a4(1,i,km))
!-----------------------------------------------------
!        a4(3,i,km) = (12./7.)*a4(1,i,km)-(13./14.)*a4(1,i,km-1)+(3./14.)*a4(1,i,km-2)
! No over- and under-shoot condition
         a4(2,i,km) = max( a4(2,i,km), min(a4(1,i,km), a4(1,i,km1)) )
         a4(2,i,km) = min( a4(2,i,km), max(a4(1,i,km), a4(1,i,km1)) )
         dc(i,km) = 0.5*(a4(1,i,km) - a4(2,i,km))
      enddo


! Enforce constraint on the "slope" at the surface

#ifdef BOT_MONO
      do i=i1,i2
         a4(4,i,km) = 0
         if( a4(3,i,km) * a4(1,i,km) <= 0. ) a4(3,i,km) = 0.
         d1 = a4(1,i,km) - a4(2,i,km)
         d2 = a4(3,i,km) - a4(1,i,km)
         if ( d1*d2 < 0. ) then
              a4(2,i,km) = a4(1,i,km)
              a4(3,i,km) = a4(1,i,km)
         else
              dq = sign(min(abs(d1),abs(d2),0.5*abs(delq(i,km-1))), d1)
              a4(2,i,km) = a4(1,i,km) - dq
              a4(3,i,km) = a4(1,i,km) + dq
         endif
      enddo
#else
      if( iv==0 ) then
          do i=i1,i2
             a4(2,i,km) = max(0.,a4(2,i,km))
             a4(3,i,km) = max(0.,a4(3,i,km))
          enddo
      elseif( iv==-1 ) then
          do i=i1,i2
             if ( a4(1,i,km)*a4(3,i,km) <= 0. ) then
                  a4(3,i,km) = 0.
             endif
          enddo
      endif
#endif

   do k=1,km1
      do i=i1,i2
         a4(3,i,k) = a4(2,i,k+1)
      enddo
   enddo

!-----------------------------------------------------------
! f(s) = AL + s*[(AR-AL) + A6*(1-s)]         ( 0 <= s  <= 1 )
!-----------------------------------------------------------
! Top 2 and bottom 2 layers always use monotonic mapping
      do k=1,2
         do i=i1,i2
            a4(4,i,k) = 3.*(2.*a4(1,i,k) - (a4(2,i,k)+a4(3,i,k)))
         enddo
         call ppm_limiters(dc(i1,k), a4(1,i1,k), it, 0)
      enddo

      if(kord >= 7) then
!-----------------------
! Huynh's 2nd constraint
!-----------------------
      do k=2,km1
         do i=i1,i2
! Method#1
!           h2(i,k) = delq(i,k) - delq(i,k-1)
! Method#2 - better
            h2(i,k) = 2.*(dc(i,k+1)/delp(i,k+1) - dc(i,k-1)/delp(i,k-1))  &
                     / ( delp(i,k)+0.5*(delp(i,k-1)+delp(i,k+1)) )        &
                     * delp(i,k)**2 
! Method#3
!!!            h2(i,k) = dc(i,k+1) - dc(i,k-1)
         enddo
      enddo

      fac = 1.5           ! original quasi-monotone

      do k=3,km-2
        do i=i1,i2
! Right edges
!        qmp   = a4(1,i,k) + 2.0*delq(i,k-1)
!        lac   = a4(1,i,k) + fac*h2(i,k-1) + 0.5*delq(i,k-1)
!
         pmp   = 2.*dc(i,k)
         qmp   = a4(1,i,k) + pmp
         lac   = a4(1,i,k) + fac*h2(i,k-1) + dc(i,k)
         a4(3,i,k) = min(max(a4(3,i,k), min(a4(1,i,k), qmp, lac)),    &
                                        max(a4(1,i,k), qmp, lac) )
! Left  edges
!        qmp   = a4(1,i,k) - 2.0*delq(i,k)
!        lac   = a4(1,i,k) + fac*h2(i,k+1) - 0.5*delq(i,k)
!
         qmp   = a4(1,i,k) - pmp
         lac   = a4(1,i,k) + fac*h2(i,k+1) - dc(i,k)
         a4(2,i,k) = min(max(a4(2,i,k),  min(a4(1,i,k), qmp, lac)),   &
                     max(a4(1,i,k), qmp, lac))
!-------------
! Recompute A6
!-------------
         a4(4,i,k) = 3.*(2.*a4(1,i,k) - (a4(2,i,k)+a4(3,i,k)))
        enddo
! Additional constraint to ensure positivity when kord=7
         if (iv == 0 .and. kord >= 6 )                      &
             call ppm_limiters(dc(i1,k), a4(1,i1,k), it, 2)
      enddo

      else

         lmt = kord - 3
         lmt = max(0, lmt)
         if (iv == 0) lmt = min(2, lmt)

         do k=3,km-2
            if( kord /= 4) then
              do i=i1,i2
                 a4(4,i,k) = 3.*(2.*a4(1,i,k) - (a4(2,i,k)+a4(3,i,k)))
              enddo
            endif
            if(kord/=6) call ppm_limiters(dc(i1,k), a4(1,i1,k), it, lmt)
         enddo
      endif

      do k=km1,km
         do i=i1,i2
            a4(4,i,k) = 3.*(2.*a4(1,i,k) - (a4(2,i,k)+a4(3,i,k)))
         enddo
         call ppm_limiters(dc(i1,k), a4(1,i1,k), it, 0)
      enddo

 end subroutine ppm_profile


 subroutine ppm_limiters(dm, a4, itot, lmt)

! !INPUT PARAMETERS:
      real , intent(in):: dm(*)     ! the linear slope
      integer, intent(in) :: itot      ! Total Longitudes
      integer, intent(in) :: lmt       ! 0: Standard PPM constraint
                                       ! 1: Improved full monotonicity constraint (Lin)
                                       ! 2: Positive definite constraint
                                       ! 3: do nothing (return immediately)
! !INPUT/OUTPUT PARAMETERS:
      real , intent(inout) :: a4(4,*)   ! PPM array
                                           ! AA <-- a4(1,i)
                                           ! AL <-- a4(2,i)
                                           ! AR <-- a4(3,i)
                                           ! A6 <-- a4(4,i)
! !LOCAL VARIABLES:
      real  qmp
      real  da1, da2, a6da
      real  fmin
      integer i

#if !defined(TAF_DOES_NOT_LIKE)
      real, parameter::  r3 = 1./3., r23 = 2./3., r12 = 1./12.
#endif

! Developer: S.-J. Lin, NASA-GSFC
! Last modified: Apr 24, 2000

      if ( lmt == 3 ) return

      if(lmt == 0) then
! Standard PPM constraint
      do i=1,itot
      if(dm(i) == 0.) then
         a4(2,i) = a4(1,i)
         a4(3,i) = a4(1,i)
         a4(4,i) = 0.
      else
         da1  = a4(3,i) - a4(2,i)
         da2  = da1**2
         a6da = a4(4,i)*da1
         if(a6da < -da2) then
            a4(4,i) = 3.*(a4(2,i)-a4(1,i))
            a4(3,i) = a4(2,i) - a4(4,i)
         elseif(a6da > da2) then
            a4(4,i) = 3.*(a4(3,i)-a4(1,i))
            a4(2,i) = a4(3,i) - a4(4,i)
         endif
      endif
      enddo

      elseif (lmt == 1) then

! Improved full monotonicity constraint (Lin 2004)
! Note: no need to provide first guess of A6 <-- a4(4,i)
      do i=1, itot
           qmp = 2.*dm(i)
         a4(2,i) = a4(1,i)-sign(min(abs(qmp),abs(a4(2,i)-a4(1,i))), qmp)
         a4(3,i) = a4(1,i)+sign(min(abs(qmp),abs(a4(3,i)-a4(1,i))), qmp)
         a4(4,i) = 3.*( 2.*a4(1,i) - (a4(2,i)+a4(3,i)) )
      enddo

      elseif (lmt == 2) then

! Positive definite constraint
      do i=1,itot
      if( abs(a4(3,i)-a4(2,i)) < -a4(4,i) ) then
      fmin = a4(1,i)+0.25*(a4(3,i)-a4(2,i))**2/a4(4,i)+a4(4,i)*r12
         if( fmin < 0. ) then
         if(a4(1,i)<a4(3,i) .and. a4(1,i)<a4(2,i)) then
            a4(3,i) = a4(1,i)
            a4(2,i) = a4(1,i)
            a4(4,i) = 0.
         elseif(a4(3,i) > a4(2,i)) then
            a4(4,i) = 3.*(a4(2,i)-a4(1,i))
            a4(3,i) = a4(2,i) - a4(4,i)
         else
            a4(4,i) = 3.*(a4(3,i)-a4(1,i))
            a4(2,i) = a4(3,i) - a4(4,i)
         endif
         endif
      endif
      enddo

      endif

 end subroutine ppm_limiters



 subroutine steepz(i1, i2, km, a4, df2, dm, dq, dp, d4)
 integer, intent(in) :: km, i1, i2
   real , intent(in) ::  dp(i1:i2,km)       ! grid size
   real , intent(in) ::  dq(i1:i2,km)       ! backward diff of q
   real , intent(in) ::  d4(i1:i2,km)       ! backward sum:  dp(k)+ dp(k-1) 
   real , intent(in) :: df2(i1:i2,km)       ! first guess mismatch
   real , intent(in) ::  dm(i1:i2,km)       ! monotonic mismatch
! !INPUT/OUTPUT PARAMETERS:
      real , intent(inout) ::  a4(4,i1:i2,km)  ! first guess/steepened
! !LOCAL VARIABLES:
      integer i, k
      real  alfa(i1:i2,km)
      real     f(i1:i2,km)
      real   rat(i1:i2,km)
      real   dg2

! Compute ratio of dq/dp
      do k=2,km
         do i=i1,i2
            rat(i,k) = dq(i,k-1) / d4(i,k)
         enddo
      enddo

! Compute F
      do k=2,km-1
         do i=i1,i2
            f(i,k) =   (rat(i,k+1) - rat(i,k))                          &
                     / ( dp(i,k-1)+dp(i,k)+dp(i,k+1) )
         enddo
      enddo

      do k=3,km-2
         do i=i1,i2
         if(f(i,k+1)*f(i,k-1)<0. .and. df2(i,k)/=0.) then
            dg2 = (f(i,k+1)-f(i,k-1))*((dp(i,k+1)-dp(i,k-1))**2          &
                   + d4(i,k)*d4(i,k+1) )
            alfa(i,k) = max(0., min(0.5, -0.1875*dg2/df2(i,k)))
         else
            alfa(i,k) = 0.
         endif
         enddo
      enddo

      do k=4,km-2
         do i=i1,i2
            a4(2,i,k) = (1.-alfa(i,k-1)-alfa(i,k)) * a4(2,i,k) +         &
                        alfa(i,k-1)*(a4(1,i,k)-dm(i,k))    +             &
                        alfa(i,k)*(a4(1,i,k-1)+dm(i,k-1))
         enddo
      enddo

 end subroutine steepz



 subroutine rst_remap(km, kn, is,ie,js,je, isd,ied,jsd,jed, nq,  &
                      delp_r, u_r, v_r, w_r, delz_r, pt_r, q_r,      &
                      delp,   u,   v,   w,   delz,   pt,   q,        &
                      ak_r, bk_r, ak, bk, hydrostatic)
!------------------------------------
! Assuming hybrid sigma-P coordinate:
!------------------------------------
! !INPUT PARAMETERS:
  integer, intent(in):: km                    ! Restart z-dimension
  integer, intent(in):: kn                    ! Run time dimension
  integer, intent(in):: nq                    ! number of tracers (including h2o)
  integer, intent(in):: is,ie,isd,ied         ! starting & ending X-Dir index
  integer, intent(in):: js,je,jsd,jed         ! starting & ending Y-Dir index
  logical, intent(in):: hydrostatic
  real, intent(in) :: ak_r(km+1)
  real, intent(in) :: bk_r(km+1)
  real, intent(in) :: ak(kn+1)
  real, intent(in) :: bk(kn+1)
  real, intent(in):: delp_r(is:ie,js:je,km) ! pressure thickness
  real, intent(in)::   u_r(is:ie,  js:je+1,km)   ! u-wind (m/s)
  real, intent(in)::   v_r(is:ie+1,js:je  ,km)   ! v-wind (m/s)
  real, intent(inout)::  pt_r(is:ie,js:je,km)
  real, intent(in)::   w_r(is:ie,js:je,km)
  real, intent(in)::   q_r(is:ie,js:je,km,*)
  real, intent(inout)::delz_r(is:ie,js:je,km)
! Output:
  real, intent(out):: delp(isd:ied,jsd:jed,kn) ! pressure thickness
  real, intent(out)::  u(isd:ied  ,jsd:jed+1,kn)   ! u-wind (m/s)
  real, intent(out)::  v(isd:ied+1,jsd:jed  ,kn)   ! v-wind (m/s)
  real, intent(out)::  w(isd:ied  ,jsd:jed  ,kn)   ! vertical velocity (m/s)
  real, intent(out):: pt(isd:ied  ,jsd:jed  ,kn)   ! temperature
  real, intent(out):: q(isd:ied,jsd:jed,kn,*)
  real, intent(out):: delz(is:ie,js:je,kn)   ! delta-height (m)
!-----------------------------------------------------------------------
  real r_vir
  real ps(isd:ied,jsd:jed)  ! surface pressure
  real  pe1(is:ie,km+1)
  real  pe2(is:ie,kn+1)
  real  pv1(is:ie+1,km+1)
  real  pv2(is:ie+1,kn+1)

  integer i,j,k , iq
  integer, parameter:: kord=4

  r_vir = rvgas/rdgas - 1.

  do j=js,je
     do i=is,ie
        ps(i,j) = ak_r(1)
     enddo
  enddo

  do k=1,km
     do j=js,je
        do i=is,ie
           ps(i,j) = ps(i,j) + delp_r(i,j,k)
        enddo
     enddo
  enddo

  call mpp_update_domains(ps, domain,  whalo=1, ehalo=1,     &
                          shalo=1, nhalo=1, complete=.true.)

! Compute virtual Temp
  do k=1,km
     do j=js,je
        do i=is,ie
           pt_r(i,j,k) = pt_r(i,j,k) * (1.+r_vir*q_r(i,j,k,1))
        enddo
     enddo
  enddo

  do 1000 j=js,je+1
!------
! map u
!------
     do k=1,km+1
        do i=is,ie
           pe1(i,k) = ak_r(k) + 0.5*bk_r(k)*(ps(i,j-1)+ps(i,j))
        enddo
     enddo

     do k=1,kn+1
        do i=is,ie
           pe2(i,k) = ak(k) + 0.5*bk(k)*(ps(i,j-1)+ps(i,j))
        enddo
     enddo

     call remap_2d(km, pe1, u_r(is:ie,j:j,1:km),       &
                   kn, pe2,   u(is:ie,j:j,1:kn),       &
                   is, ie, -1, kord)

  if ( j /= (je+1) )  then 

!---------------
! Hybrid sigma-p
!---------------
     do k=1,km+1
        do i=is,ie
           pe1(i,k) = ak_r(k) + bk_r(k)*ps(i,j)
        enddo
     enddo

     do k=1,kn+1
        do i=is,ie
           pe2(i,k) =   ak(k) + bk(k)*ps(i,j)
        enddo
     enddo

!-------------
! Compute delp
!-------------
      do k=1,kn
         do i=is,ie
            delp(i,j,k) = pe2(i,k+1) - pe2(i,k)
         enddo
      enddo

!----------------
! Map constituents
!----------------
      if( nq /= 0 ) then
          do iq=1,nq
             call remap_2d(km, pe1, q_r(is:ie,j:j,1:km,iq:iq),  &
                           kn, pe2,   q(is:ie,j:j,1:kn,iq:iq),  &
                           is, ie, 0, kord)
          enddo
      endif

      if ( .not. hydrostatic ) then
! Remap vertical wind:
         call remap_2d(km, pe1, w_r(is:ie,j:j,1:km),       &
                       kn, pe2,   w(is:ie,j:j,1:kn),       &
                       is, ie, -1, kord)
! Remap delz for hybrid sigma-p coordinate
         do k=1,km
            do i=is,ie
               delz_r(i,j,k) = -delz_r(i,j,k)/delp_r(i,j,k) ! ="specific volume"/grav
            enddo
         enddo
         call remap_2d(km, pe1, delz_r(is:ie,j:j,1:km),       &
                       kn, pe2,   delz(is:ie,j:j,1:kn),       &
                       is, ie, 1, kord)
         do k=1,kn
            do i=is,ie
               delz(i,j,k) = -delz(i,j,k)*delp(i,j,k)
            enddo
         enddo
      endif

! Geopotential conserving remap of virtual temperature:
       do k=1,km+1
          do i=is,ie
             pe1(i,k) = log(pe1(i,k))
          enddo
       enddo
       do k=1,kn+1
          do i=is,ie
             pe2(i,k) = log(pe2(i,k))
          enddo
       enddo

       call remap_2d(km, pe1, pt_r(is:ie,j:j,1:km),       &
                     kn, pe2,   pt(is:ie,j:j,1:kn),       &
                     is, ie, 1, kord)
!------
! map v
!------
       do k=1,km+1
          do i=is,ie+1
             pv1(i,k) = ak_r(k) + 0.5*bk_r(k)*(ps(i-1,j)+ps(i,j))
          enddo
       enddo
       do k=1,kn+1
          do i=is,ie+1
             pv2(i,k) = ak(k) + 0.5*bk(k)*(ps(i-1,j)+ps(i,j))
          enddo
       enddo

       call remap_2d(km, pv1, v_r(is:ie+1,j:j,1:km),       &
                     kn, pv2,   v(is:ie+1,j:j,1:kn),       &
                     is, ie+1, -1, kord)

  endif !(j < je+1)
1000  continue

  do k=1,kn
     do j=js,je
        do i=is,ie
           pt(i,j,k) = pt(i,j,k) / (1.+r_vir*q(i,j,k,1))
        enddo
     enddo   
  enddo

 end subroutine rst_remap



 subroutine mappm(km, pe1, q1, kn, pe2, q2, i1, i2, iv, kord)

! IV = 0: constituents
! IV = 1: potential temp
! IV =-1: winds
 
! Mass flux preserving mapping: q1(im,km) -> q2(im,kn)
 
! pe1: pressure at layer edges (from model top to bottom surface)
!      in the original vertical coordinate
! pe2: pressure at layer edges (from model top to bottom surface)
!      in the new vertical coordinate

 integer, intent(in):: i1, i2, km, kn, kord, iv
 real, intent(in ):: pe1(i1:i2,km+1), pe2(i1:i2,kn+1)
 real, intent(in )::  q1(i1:i2,km)
 real, intent(out)::  q2(i1:i2,kn)
! local
      real dp1(i1:i2,km)
      real a4(4,i1:i2,km)
      integer i, k, l
      integer k0, k1
      real pl, pr, tt, delp, qsum, dpsum, esl

#if !defined(TAF_DOES_NOT_LIKE)
      real, parameter::  r3 = 1./3., r23 = 2./3., r12 = 1./12.
#endif

      do k=1,km
         do i=i1,i2
             dp1(i,k) = pe1(i,k+1) - pe1(i,k)
            a4(1,i,k) = q1(i,k)
         enddo
      enddo

      if ( kord >7 ) then
           call  cs_profile( a4, dp1, km, i1, i2, iv )
      else
           call ppm_profile( a4, dp1, km, i1, i2, iv, kord )
      endif

!------------------------------------
! Lowest layer: constant distribution
!------------------------------------
      do i=i1,i2
         a4(2,i,km) = q1(i,km)
         a4(3,i,km) = q1(i,km)
         a4(4,i,km) = 0.
      enddo

      do 5555 i=i1,i2
         k0 = 1
      do 555 k=1,kn

         if(pe2(i,k+1) .le. pe1(i,1)) then
! Entire grid above old ptop
            q2(i,k) = a4(2,i,1)
         elseif(pe2(i,k) .ge. pe1(i,km+1)) then
! Entire grid below old ps
            q2(i,k) = a4(3,i,km)
         elseif(pe2(i,k  ) .lt. pe1(i,1) .and.   &
                pe2(i,k+1) .gt. pe1(i,1))  then
! Part of the grid above ptop
            q2(i,k) = a4(1,i,1)
         else

         do 45 L=k0,km
! locate the top edge at pe2(i,k)
         if( pe2(i,k) .ge. pe1(i,L) .and.        &
             pe2(i,k) .le. pe1(i,L+1)    ) then
             k0 = L
             PL = (pe2(i,k)-pe1(i,L)) / dp1(i,L)
             if(pe2(i,k+1) .le. pe1(i,L+1)) then

! entire new grid is within the original grid
               PR = (pe2(i,k+1)-pe1(i,L)) / dp1(i,L)
               TT = r3*(PR*(PR+PL)+PL**2)
               q2(i,k) = a4(2,i,L) + 0.5*(a4(4,i,L)+a4(3,i,L)  &
                       - a4(2,i,L))*(PR+PL) - a4(4,i,L)*TT
              goto 555
             else
! Fractional area...
              delp = pe1(i,L+1) - pe2(i,k)
              TT   = r3*(1.+PL*(1.+PL))
              qsum = delp*(a4(2,i,L)+0.5*(a4(4,i,L)+            &
                     a4(3,i,L)-a4(2,i,L))*(1.+PL)-a4(4,i,L)*TT)
              dpsum = delp
              k1 = L + 1
             goto 111
             endif
         endif
45       continue

111      continue
         do 55 L=k1,km
         if( pe2(i,k+1) .gt. pe1(i,L+1) ) then

! Whole layer..

            qsum  =  qsum + dp1(i,L)*q1(i,L)
            dpsum = dpsum + dp1(i,L)
         else
           delp = pe2(i,k+1)-pe1(i,L)
           esl  = delp / dp1(i,L)
           qsum = qsum + delp * (a4(2,i,L)+0.5*esl*            &
                 (a4(3,i,L)-a4(2,i,L)+a4(4,i,L)*(1.-r23*esl)) )
          dpsum = dpsum + delp
           k0 = L
           goto 123
         endif
55       continue
        delp = pe2(i,k+1) - pe1(i,km+1)
        if(delp > 0.) then
! Extended below old ps
           qsum = qsum + delp * a4(3,i,km)
          dpsum = dpsum + delp
        endif
123     q2(i,k) = qsum / dpsum
      endif
555   continue
5555  continue

 end subroutine mappm


end module fv_mapz_mod
